%* lsaf_validate_sasdataset_name.sas                                                   *;
%*                                                                                     *;
%* Validates the data set that is passed by the user to the SAS macros. The tests      *;
%* are:                                                                                *;
%* - The data set has a valid SAS name.                                                *;
%*                                                                                     *;
%* Syntax:                                                                             *;
%* %LSAF_VALIDATE_SASDATASET_NAME(SAS_DSN=SAS-data-set                                 *;
%*            <, CALLINGMACRO=calling-macro>);                                         *;
%*                                                                                     *;
%* macvar _lsafRC_  The return code.                                                   *;
%* macvar _lsafMsg_ The return message.                                                *;
%*                                                                                     *;
%* param sas_dsn - Required - The name of the data set to validate.                    *;
%* param callingMacro - Optional - The macro that called this macro.                   *;
%* param dsType - Optional - The type of data set: Input or Output(default)            *;
%*                                                                                     *;
%* macrotype LSAF                                                                      *;
%* since 2.5                                                                           *;
%* exposure internal                                                                   *;
%*                                                                                     *;
%* History                                                                             *;
%* 2021-04-19  pulled out of lsaf_validate_sasdataset                                  *;
%* 2021-08-18  Update header documentation                                             *;
%*                                                                                     *;

%macro lsaf_validate_sasdataset_name(
   sas_dsn=,
   callingMacro=,
   dsType=,
   dataset_must_exist=0
   ) / des='Validates the name of SAS data set used for input or output of a SAS macro.';

   %if ("&dsType" eq "") %then %let dsType=Output;
   %let __errorMsg__=%str(&dsType. data set name is not valid.);

   %put NOTE: &callingMacro.: Validating data set: &sas_dsn;

   %****************************************;
   %* Initialize macro variables           *;
   %****************************************;
   %global _lsafMsg_
           _lsafRC_
   ;
   %local validatingNotesOption;

   %let _lsafRC_=%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

   %****************************************;
   %* VERIFY: data set name                *;
   %****************************************;
   %if (%qcmpres(&sas_dsn) eq %str()) %then
   %do;	  
      %let _lsafMsg_= %str(&dsType data set name was not provided.);
      %return;
   %end;

   %let validatingNotesOption=%sysfunc(getoption(notes));
   options nonotes;

   %let dsnMsg=;

   data _null_;
      length message $45 libName $8 dsName $300;
      providedDatasetName=kstrip("&sas_dsn");
      dsName="INVALID";
      dsNameIsValid=0;

      %************************************************************************;
      %* Check for 2 level data set name (ie work.ABC).                       *;
      %* If 0 then one level name (ie ABC).                                   *;
      %* If 1 then it is a two level name (ie work.ABC).                      *;
      %* If 2 or more periods then an invalid data set name (ie work.xyz.ABC) *;
      %************************************************************************;
      dotCount=countc(providedDatasetName,".");
      dotPosition=kindexc(providedDatasetName,".");
      *put dotPosition=;
      %***************************************************;
      %* Check to make sure we have a complete two level *;
      %* data set name (ie .ABC and work. are invalid).  *;
      %***************************************************;
      if (dotPosition=1 or dotPosition=length(providedDatasetName)) then 
      do;
        message=" Missing either LIBNAME or data set.";
		GOTO SETVARS;
      end;

      select (dotCount);
	  	when (0) dsName=providedDatasetName;
        when (1) do;
           %*******************************************************;
           %* check if libname statement longer than 8 characters *;
           %*******************************************************;
           if (dotPosition le 9) then
		   do;
		      libName = (ksubstr(providedDatasetName, 1, dotPosition-1));
              *put libName=;
              dsName = (ksubstr(providedDatasetName, dotPosition+1));
		   end;
           else 
           do;
              message="Libname longer than 8 characters.";
			  GOTO SETVARS;
           end;
		end;
        otherwise do; 
		   	  message="Invalid data set name structure.";
		   	  GOTO SETVARS;
        end;		   
      end;

      firstchar=ksubstr(kleft(dsName), 1, 1);
      if (anydigit(firstchar)) then do;
		message="Data set name must start with a character.";
		GOTO SETVARS;
      end;

      if (length(dsName) > 32) then do;
        message="Data set name longer than 32 characters.";
		GOTO SETVARS;
      end;

	dsNameIsValid=1;

	SETVARS:
        call symput("_lsafvaliddsn_", strip(put(dsNameIsValid, best.)));
        call symput("dsnMsg", message);
		call symput("specifiedLib", strip(libName));
   run;

   options &validatingNotesOption;

   %if (&_lsafvaliddsn_. eq 0) %then
   %do;
     %let _lsafMsg_= &__errorMsg__. &dsnMsg;
	 %return;
   %end;
 
   %let haveLibRef = %eval("&specifiedLib." ne "");
   %if (&haveLibRef.) %then
   %do;
      %let libIsAssignedRC = %sysfunc(libref(&specifiedLib.));
 	  %if (&libIsAssignedRC.) %then %do;
	 	 %lsaf_set_lsafmsg_from_sysmsg(%str(%sysfunc(sysmsg())));
	 	 %return;
	  %end;
   %end;

   * CHECK EXISTS: Also getting errors for data set name.;
   %let _dsnExists = %sysfunc(exist(&sas_dsn.));
   %let _dsnExistsMsg = %sysfunc(sysmsg());

   %if (&_dsnExists. eq 1) %then
   	  %GOTO ENDVALIDATION; 

   * DATA SET DOES NOT EXIST ;

   %if (&dataset_must_exist. eq 1) %then 
   %do;
     %let _lsafMsg_= %str(&dsType data set &sas_dsn does not exist. );
	 %return;
   %end;   

   %*put &=_dsnExistsMsg;
   %let hasNotExist = %index(&_dsnExistsMsg, %str(not exist));
   %*put &=hasNotExist;
   %if (&hasNotExist. ge 1) %then
       %GOTO ENDVALIDATION; 
 
   %lsaf_set_lsafmsg_from_sysmsg(%str(&_dsnExistsMsg.));

   %return;

   %ENDVALIDATION:

   %let _lsafRC_=0;

%mend lsaf_validate_sasdataset_name;
