%* lsaf_validate_sasdataset.sas                                                        *;
%*                                                                                     *;
%* Validates the data set that is passed by the user to the SAS macros. The tests      *;
%* are:                                                                                *;
%* - The data set has a valid SAS name.                                                *;
%* - The data set exists.                                                              *;
%* - Optional (Default) - The data set is not empty (has observations).                *;
%* - The data set includes the required variables.                                     *;
%* - The required variables are of the correct variable type (character or numeric)    *;                                                       *;
%*                                                                                     *;
%* Syntax:                                                                             *;
%* %LSAF_VALIDATE_SASDATASET(SAS_DSN=SAS-data-set                                      *;
%*            <, CALLINGMACRO=calling-macro, REQUIRED_VARIABLES=required-var-list,     *;
%*            REQUIRED_NUMERIC_VARIABLES=required-numeric-list>,                       *;
%*            ALLOW_EMPTY_DATASETS=allow-empty-datasets);                              *;
%*                                                                                     *;
%* macvar _lsafRC_  The return code.                                                   *;
%* macvar _lsafMsg_ The return message.                                                *;
%*                                                                                     *;
%* param sas_dsn - Required - The name of the data set to validate.                    *;
%* param callingMacro - Optional - The macro that called this macro.                   *;
%* param required_variables - Optional - A space-delimited list of data set variables  *;
%*             that are required to be present in the data set.                        *;
%* param required_numeric_variables - Optional - A space-delimited list of the         *;
%*             required variables that must be of type numeric. If a required          *;
%*             variable is not included in this list, it is validated as a             *;
%*             character.                                                              *;
%* param allow_empty_datasets - Optional - Indicates whether an error is returned      *;
%*             when the input data set has 0 obs. The default is 0 (do not allow       *;
%*             0 obs).                                                                 *;
%* param dataset_must_exist - Optional - Indicates whether the data set must exist.    *;
%*             Used when data set does not need to exist ahead of time.                *;
%*             Default is 1 (must exist).                                              *;
%*                                                                                     *;
%* macrotype LSAF                                                                      *;
%* since 1.2                                                                           *;
%* exposure internal                                                                   *;
%*                                                                                     *;
%* History                                                                             *;
%* 2013-08-30  updates for standardization and coding conventions.                     *;
%* 2014-03-28  decoupled from updateacls.                                              *;
%* 2016-08-26  rebrand as LSAF                                                         *;
%* 2016-11-07  added parameter allow_empty_datasets                                    *;
%* 2017-01-09  updated documentation for allow_empty_datasets                          *;
%* 2021-01-20  corrected misspellings                                                  *;
%* 2021-03-24  added option validate_name_only                                         *;
%* 2021-04-19  pulled out tests for data set name,                                     *;
%*             changed validate_name_only to datset_must_exist                         *; 
%* 2021-08-18  Update header documentation                                             *;
%*                                                                                     *;

%macro lsaf_validate_sasdataset(
   sas_dsn=,
   callingMacro=,
   required_variables=,
   required_numeric_variables=%str(),
   allow_empty_datasets=0,
   dataset_must_exist=1
   ) / des='Validates an input SAS data set.';

   %**************************************************************;
   %* Validate the data set name.                                *; 
   %* Also initializes global macro vars _lsafRC_ and _lsafMsg_. *;
   %**************************************************************;

   %lsaf_validate_sasdataset_name(sas_dsn=&sas_dsn., callingMacro=&callingMacro., 
		dsType=Input, dataset_must_exist=&dataset_must_exist.);

   %*put &SYSMACRONAME.: &=_lsafRC_.;

   %if (&_lsafRC_ ne 0) %then 
      %return;

   %let _lsafRC_=%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

   %local dsid;
   %let DSID=%sysfunc(OPEN(&sas_dsn., IN));

   %****************************************;
   %* VERIFY: data set has observations    *;
   %****************************************;
   %if (&allow_empty_datasets EQ 0) %then
   %do;
     %local nobs dsrc;
     %let RECORDS=%sysfunc(ATTRN(&DSID, NLOBS));

     %if (%eval(&RECORDS eq 0)) %then
     %do;

       %let _lsafMsg_= %str(Input data set &sas_dsn has no observations.);
       %let dsrc=%sysfunc(CLOSE(&DSID));
       %return;
     %end;
   %end;


   %********************************************************************;
   %* VERIFY: required variables exist and are of the correct type.
   %********************************************************************;
   %if (%qcmpres(&required_variables) eq %str()) 
     %then %GOTO END_VALIDATION;

   %local varCnt varName varPos missingVars invalidVars DSINVALID;
   %let varCnt=1;
   %let varName=_null_;
   %let missingVars=%str();
   %let invalidCharVars=%str();
   %let invalidNumVars=%str();
   %let DSINVALID=0;

   %do %until (&varName=%str());
     %let varName=%scan(&required_variables, &varCnt);
     %if (&varName eq %str()) 
       %then %goto CONTINUE;
     %let varPos=%sysfunc(varnum(&DSID,&varName));

     %************************************;
     %* variable is missing              *;
     %************************************;
     %if (%eval(&varPos < 1)) %then
     %do;
       %let missingVars=%str(%trim(&missingVars) %trim(&varName));
       %let DSINVALID=1;
       %goto CONTINUE;
     %end;

     %let actualVarType = %sysfunc( vartype(&DSID, &varPos) );

     %if (%index(&required_numeric_variables, &varName) > 0) %then
     %do;
       %******************************************;
       %* variable that should be NUMERIC is not *;
       %******************************************;
       %if (&actualVarType ne N) %then
       %do;
         %let invalidNumVars=%str(%trim(&invalidNumVars) %trim(&varName));
         %let DSINVALID=1;
       %end;
     %end;
     %else %if (&actualVarType ne C) %then
     %do;
       %*********************************************;
       %*  variable that should be CHARACTER is not *;
       %*********************************************;
       %let invalidCharVars=%str(%trim(&invalidCharVars) %trim(&varName));
       %let DSINVALID=1;
     %end;

     %CONTINUE:
     %let varCnt=%eval(&varCnt+1);
   %end;

   %let rc=%sysfunc(CLOSE(&DSID));

   %if (&DSINVALID ne 0) %then
   %do;
     %**************************************************************************************;
     %* only setting one message so giving missing variables precedence over variable type *;
     %**************************************************************************************;
     %if (&missingVars ne %str()) %then
     %do;
       %let _lsafMsg_= %cmpres(Data set &sas_dsn is missing required variables: &missingVars);
       %put;
     %end;
     %else %if (&invalidNumVars ne %str()) %then
     %do;
       %let _lsafMsg_= %cmpres(Variables in data set &sas_dsn that must be numeric are not: &invalidNumVars);
     %end;
     %else %if (&invalidCharVars ne %str()) %then
     %do;
       %let _lsafMsg_= %cmpres(Variables in data set &sas_dsn that must be character are not: &invalidCharVars);
     %end;
     %return;
   %end;

   %END_VALIDATION:
   %if (&dsid ne 0) %then %let rc=%sysfunc(CLOSE(&DSID));
   %let _lsafRC_=%str(0);
   %let _lsafMsg_=%str(Data set &sas_dsn is valid for &callingMacro..);

%mend lsaf_validate_sasdataset;
