%* lsaf_validate_parameters_dataset                                                                  *;
%*                                                                                                   *;
%* Validates the parameters data sets used to submit a job.                                          *;
%*                                                                                                   *;
%* Syntax:                                                                                           *;
%* %LSAF_VALIDATE_PARAMETERS_DATASET(SAS_DSN=SAS-data-set <, CALLINGMACRO=calling-macro> );          *;
%*                                                                                                   *;
%* macvar _lsafRC_  the return code.                                                                 *;
%* macvar _lsafMsg_ the return message.                                                              *;
%*                                                                                                   *;
%* macrotype LSAF                                                                                    *;
%* since 1.5                                                                                         *;
%* exposure internal                                                                                 *;
%*                                                                                                   *;
%* History                                                                                           *;
%* 2016-10-25  initial coding                                                                        *;
%* 2019-02-25  update to 2.1                                                                         *;
%* 2021-08-18  Update header documentation                                                           *;
%*                                                                                                   *;

%macro lsaf_validate_parameters_dataset(
   callingMacro=,
   sas_dsn=
   ) / des='Validates a SAS data sets for use in parameter substitution when submitting a job.';

   %put NOTE: &callingMacro.: Validating dataset &sas_dsn;

   %****************************************;
   %* Initialize macro variables           *;
   %****************************************;
   %global _lsafMsg_
           _lsafRC_
   ;

   %let _lsafRC_=%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

   %*************************************************************;
   %* Turn off the notes generated for all the pre-processing.  *;
   %*************************************************************;
   options nonotes;

   %***************************************************************************;
   %* Cleanup from any previous runs that may have left temp data set behind. *;
   %***************************************************************************;
   %let _tempSubmitParameters_tempDS_=%str(__SASMacro_submitJobtemp__);

   %if (%sysfunc(exist(&_tempSubmitParameters_tempDS_)) > 0) %then
   %do;
      %put NOTE: &macroName: Deleting temporary data set &_tempSubmitParameters_tempDS_ before processing.;
      proc datasets library=work nolist;
         delete &_tempSubmitParameters_tempDS_;
      quit;
      run;
   %end;

   %***************************************************************************;
   %* Validate the parameters data set.                                       *;
   %* sets global macro variable _lsafRC_=0 if validation successful          *;
   %***************************************************************************;
   %lsaf_validate_sasdataset(callingMacro=&macroName,
                                   sas_dsn=&sas_dsn,
                                   required_variables=%str(name type value fileVersion, includeSubFolders),
                                   required_numeric_variables=%str(includeSubFolders)
   );
   %if (&_lsafRC_ ne 0) %then
   %do;
      %return;
   %end;

   %*****************************************************************;
   %* Validate that no duplicate values are present for parameters. *;
   %* Order of original data set needs to be preserved.             *;
   %*****************************************************************;
   %PUT NOTE: TESTING FOR DUPLICATE PARAMETER VALUES.;

   %let _lsafRC_ =%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

   proc sort data=&sas_dsn out=&_tempSubmitParameters_tempDS_;
      by name;
   run;
   
   data _null_;
      set &_tempSubmitParameters_tempDS_;
      by name;
      if (first.name AND NOT(last.name)) then
      do;
         call symput("_lsafMsg_", "Input data set has duplicate values for parameter name " || name || ".");
         STOP;
      end;
      call symput("_lsafRC_", "0");
   run;

   option notes;

%mend lsaf_validate_parameters_dataset;