%* lsaf_updateuserproperties                                                                 *;
%*                                                                                           *;
%* Updates the editable properties of a user account using a SAS data set as input.          *;
%*                                                                                           *;
%* macrotype LSAF                                                                            *;
%* since 2.3                                                                                 *;
%* exposure external                                                                         *;
%*                                                                                           *;
%* History                                                                                   *;
%* 2019-12-17  initial coding                                                                *;
%* 2020-12-04  updates for service refactor                                                  *;
%* 2021-08-18  Remove header documentation                                                   *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                            *;
%*                                                                                           *;

%macro lsaf_updateuserproperties(
   lsaf_userid=,
   sas_dsname=
   ) /des='Updates properties of a user based on the input data set';

   %****************************************;
   %* Initialize macro variables           *;
   %****************************************;
   %lsaf_getProductName();

   %global _lsafMsg_
           _lsafRC_
   ;
   %local logMsgPrefix
          logMsgPrimary
          macroName
          notesOption
          _tempUPDATEPROPS_tempDS_
          tempDsMsg
   ;

   %let _lsafRC_ =%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

   %let macroName=%sysfunc(lowcase(&SYSMACRONAME));
   %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
   %let logMsgPrefix=NOTE: &_LSAF_PRODUCT_NAME_ Macro: *;

   %*************************************************************;
   %* Turn off the notes generated for all the pre-processing.  *;
   %* Save original option setting so it can be reset properly. *;
   %*************************************************************;
   %let notesOption=%sysfunc(getoption(notes));
   options nonotes;

   %***************************************************************************;
   %* Cleanup from any previous runs that may have left temp data set behind. *;
   %***************************************************************************;
   %let _tempUPDATEPROPS_tempDS_=%str(__SASMacro_updatepropstemp__);
   %let tempDsMsg=%str(Temporary sorted data set work.&_tempUPDATEPROPS_tempDS_ will not be created.);

   %if (%sysfunc(exist(&_tempUPDATEPROPS_tempDS_)) > 0) %then
   %do;
      options &notesOption.;
      %put NOTE: &macroName: Deleting temporary data set &_tempUPDATEPROPS_tempDS_ before processing.;
	  option nonotes;
      proc datasets library=work nolist;
         delete &_tempUPDATEPROPS_tempDS_;
         quit; 
      run;
   %end;

  %***************************************************************************;
  %* Validate the properties data set                                        *;
  %* sets global macro variable _lsafRC_=0 if validation successful          *;
  %***************************************************************************;
  %lsaf_validate_sasdataset(callingMacro=&macroName,
                            sas_dsn=&sas_dsname,
                            required_variables=%str(name value)
  );
  %if (&_lsafRC_ ne 0) %then
  %do;
     %GOTO PRINT_DSERROR_LOG_MSGS;
  %end;

  %*****************************************************************;
  %* Validate that no duplicate values are present for properties. *;
  %* Order of original data set will be preserved.                 *;
  %*****************************************************************;
  %PUT NOTE: TESTING FOR DUPLICATE PROPERTIES VALUES.;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  proc sort data=&sas_dsname out=&_tempUPDATEPROPS_tempDS_;
     by name;
  run;
  
  data _null_;
     set &_tempUPDATEPROPS_tempDS_;
     by name;
     if (first.name AND NOT(last.name)) then
     do;
        call symput("tempDsMsg", "Temporary sorted data set work.&_tempUPDATEPROPS_tempDS_ was created.");
        call symput("_lsafMsg_", "Input data set has duplicate values for property name.");
        STOP;
     end;
     call symput("_lsafRC_", "0");
  run;
  
  options &notesOption.;

  %if (&_lsafRC_ ne 0) %then
  %do;
     %GOTO PRINT_DSERROR_LOG_MSGS;
  %end;

  %**********************************************;
  %* Process the records                        *;
  %**********************************************;
  data _null_;
    attrib returnCode               length=8;
    attrib message                  length=$200;

    set &_tempUPDATEPROPS_tempDS_ end=eof;

    %***********************************;
    %* Declare the java objects, once  *;
    %***********************************;
    if (_n_ = 1) then
    do;
      declare javaobj props("com/sas/lsaf/macro/common/SasProperties");
      declare javaobj srv("com/sas/lsaf/macro/security/user/UserService");
      declare javaobj results("com/sas/lsaf/macro/common/result/Result");

      put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * " %sysfunc(lowcase("&SYSMACRONAME"));
      put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * Input data set = &sas_dsname";
    end;
    props.callVoidMethod("addProperty", kstrip(name), kstrip(value));

   if (eof) then
   do;
      %***********************************;
      %* Update the properties           *;
      %***********************************;
      srv.callStaticVoidMethod("updateProperties", "%bquote(&lsaf_userid)", props, results);

      %***********************************;
      %* Retrieve the results            *;
      %***********************************;
      results.callIntMethod("getReturnCode", returnCode);
      results.callStringMethod("getReturnMessage", message);

      %***********************************;
      %* Process the results             *;
      %***********************************;
      logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
      if (returnCode ne 0) then
      do;
         logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
         logMsg=" Cannot update user &lsaf_userid.";
      end;
      else
      do;
         logMsgPrimary=logMsgPrefix;
         logMsg=" Updated user &lsaf_userid.";
      end;

      %***********************************;
      %* Print messages in log           *;
      %***********************************;
      put;
      put logMsgPrimary " " logMsg;
      put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
      put logMsgPrefix " _lsafMsg_= " message ;
      put logMsgPrefix " _lsafRC_= " returnCode ;
      put;

      %***********************************;
      %* Set the macro variables         *;
      %***********************************;
      call symputx("_lsafRC_", returnCode);
      call symputx("_lsafMsg_", message);
   end;

   run;

   %return;

   %PRINT_DSERROR_LOG_MSGS:
      options &notesOption.;
      %put &logMsgPrimary _lsafMsg_= &_lsafMsg_ ;
      %put &logMsgPrefix  &macroName ;
      %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;
      %put &logMsgPrefix &tempDsMsg ;
      %put;

%mend lsaf_updateuserproperties;
