%* lsaf_updatestudytlfproperties                                                          *;
%*                                                                                        *;
%* Updates the editable properties of a study tlf using a SAS data set as input.          *;
%*                                                                                        *;
%* macrotype LSAF                                                                         *;
%* since 2.5                                                                              *;
%* exposure external                                                                      *;
%*                                                                                        *;
%* History                                                                                *;
%* 2021-03-25  initial coding                                                             *;
%* 2021-08-17  Remove header documentation                                                *;
%*                                                                                        *;

%macro lsaf_updatestudytlfproperties(
  lsaf_studyId=, 
  lsaf_basestandardname=,
  lsaf_modelId=,
  lsaf_tlfIdentifier=,
  sas_dsname=
  ) /des='Updates properties of a tlf on the input data set';

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;
  %local logMsgPrefix
         logMsgPrimary
         macroName
         notesOption
         tempDsMsg
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %let macroName=%sysfunc(lowcase(&SYSMACRONAME));
  %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
  %let logMsgPrefix=NOTE: &_LSAF_PRODUCT_NAME_ Macro: *;

  %*************************************************************;
  %* Save original option setting so it can be reset properly. *;
  %*************************************************************;
  %let notesOption=%sysfunc(getoption(notes));

  %***************************************************************************;
  %* Validate the properties data set                                        *;
  %* sets global macro variable _lsafRC_=0 if validation successful          *;
  %***************************************************************************;
  %lsaf_validate_sasdataset(callingMacro=&macroName,
                            sas_dsn=&sas_dsname,
                            required_variables=%str(sasname value valuequalifier)
  );
  %if (&_lsafRC_ ne 0) %then
  %do;
     %GOTO PRINT_DSERROR_LOG_MSGS;
  %end;

  options &notesOption.;

  %**********************************************;
  %* Process the records                        *;
  %**********************************************;
  data _null_;
    attrib returnCode               length=8;
    attrib message                  length=$200;

    set &sas_dsname end=eof;

    %***********************************;
    %* Declare the java objects, once  *;
    %***********************************;
    if (_n_ = 1) then
    do;
      declare javaobj props("com/sas/lsaf/macro/clinical/study/tlf/SasQualifiedProperties");
      declare javaobj srv("com/sas/lsaf/macro/clinical/study/tlf/StudyTlfService") ;
      declare javaobj results("com/sas/lsaf/macro/common/result/Result");
      put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * " %sysfunc(lowcase("&SYSMACRONAME"));
      put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * Input data set = &sas_dsname";
    end;
    props.callVoidMethod("addProperty", kstrip(sasname), kstrip(value), kstrip(valueQualifier));

    if (eof) then
    do;
      %***********************************;
      %* Update the properties           *;
      %***********************************;
      srv.callStaticVoidMethod("updateTlfProperties", "%bquote(&lsaf_studyId)", "%bquote(&lsaf_baseStandardName)", "%bquote(&lsaf_modelId)",
                               "%bquote(&lsaf_tlfIdentifier)", props, results);

      %***********************************;
      %* Retrieve the results            *;
      %***********************************;
      results.callIntMethod("getReturnCode", returnCode);
      results.callStringMethod("getReturnMessage", message);

      %***********************************;
      %* Process the results             *;
      %***********************************;
      logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
      if (returnCode ne 0) then
      do;
        logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
        logMsg=" Cannot update study tlf &lsaf_tlfIdentifier.";
      end;
      else
      do;
        logMsgPrimary=logMsgPrefix;
        logMsg=" Updated study tlf &lsaf_tlfIdentifier.";
      end;

      %***********************************;
      %* Print messages in log           *;
      %***********************************;
      put;
      put logMsgPrimary " " logMsg;
      put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
      put logMsgPrefix " _lsafMsg_= " message ;
      put logMsgPrefix " _lsafRC_= " returnCode ;
      put;

      %***********************************;
      %* Set the macro variables         *;
      %***********************************;
      call symputx("_lsafRC_", returnCode);
      call symputx("_lsafMsg_", message);
    end;

  run;

  %return;

  %PRINT_DSERROR_LOG_MSGS:
  options &notesOption;
  %put &logMsgPrimary _lsafMsg_= &_lsafMsg_ ;
  %put &logMsgPrefix  &macroName ;
  %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;
  %put &logMsgPrefix &tempDsMsg ;
  %put;

%mend lsaf_updatestudytlfproperties;
