%* lsaf_updatestudydictionaries                                                                     *;
%*                                                                                                  *;
%* Updates the controlled terminology dictionaries for a study using a SAS data set as input.       *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.2                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2019-08-14  initial coding                                                                       *;
%* 2021-08-16  Remove header documentation                                                          *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                                   *;
%*                                                                                                  *;

%macro lsaf_updatestudydictionaries(lsaf_studyid=,  
  sas_dsname=
  ) /des='Update the controlled terminology dictionaries for a SAS Life Science Analytics Framework study.';

  %*************************************************************;
  %* Turn off the notes generated for all the pre-processing.  *;
  %* Save original option setting so it can be reset properly. *;
  %*************************************************************;
  %let notesOption=%sysfunc(getoption(notes));

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local logMsgPrefix
         logMsgPrimary
         macroName
		 notesOption
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let macroName=%sysfunc(lowcase(&SYSMACRONAME));
  %let notesOption=%sysfunc(getoption(notes));
  
  %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
  %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );

  %***************************************************************************;
  %* Validate the input data set                                             *;
  %* sets global macro variable _lsafRC_=0 if validation successful          *;
  %***************************************************************************;
  %lsaf_validate_sasdataset(callingMacro=&macroName,
        sas_dsn=&sas_dsname,
        required_variables=%str(id)
        );
  options &notesOption.;

  %if (&_lsafRC_ ne 0) %then
  %do;
    %put &logMsgPrimary _lsafMsg_= &_lsafMsg_ ;
    %put &logMsgPrefix &macroName ;
    %put &logMsgPrefix Input data set = &sas_dsname;
    %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;
    %put;
    %return;
  %end;

  %***************************************************************************;
  %* If data set is valid, process it.                                       *;
  %***************************************************************************;
  %put NOTE: &macroName: Processing data set &sas_dsname;

  %***********************;
  %* Process the records *;
  %***********************;
  data _null_;
    attrib returnCode    length=8
           message       length=$200
           logMsgPrefix  length=$70
           logMsgPrimary length=$70
           logMsg        length=$300
    ;
    set &sas_dsname end=eof;

    %**********************************;
    %* Declare the java object, once  *;
    %**********************************;
    if (_n_ = 1) then
    do;
      declare javaobj dictionaryIds("com/sas/lsaf/macro/common/SasStrings");
    end;

    %*************************************;
    %* Populate the dictionaryIds list   *; 
    %*************************************;
    dictionaryIds.callVoidMethod("addValue", kstrip(id));

    if eof then 
    do;
      %***********************************;
      %* Call the update                 *;
      %***********************************; 
      declare javaobj srv("com/sas/lsaf/macro/clinical/study/StudyService");
      declare javaobj results("com/sas/lsaf/macro/common/result/Result");
      srv.callStaticVoidMethod("setStudyDictionaries", "%bquote(&lsaf_studyid)", dictionaryIds, results);

      %***********************************;
      %* Retrieve the results            *;
      %***********************************;
      results.callIntMethod("getReturnCode", returnCode);
      results.callStringMethod("getReturnMessage", message);

      %***********************************;
      %* Process the results             *;
      %***********************************;
      logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
      if (returnCode ne 0) then
      do;
        logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
        logMsg=" Cannot update controlled terminology dictionaries.";
      end;
      else
      do;
        logMsgPrimary=logMsgPrefix;
        logMsg=" Update of controlled terminology dictionaries was performed.";
      end;

      %***********************************;
      %* Print messages in log           *;
      %***********************************;
      put;
      put logMsgPrimary " " logMsg;
      put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
      put logMsgPrefix " _lsafMsg_= " message ;
      put logMsgPrefix " _lsafRC_= " returnCode ;
      put logMsgPrefix " Input data set = &sas_dsname";
      put;

      %***********************************;
      %* Set the macro variables         *;
      %***********************************;
      call symputx("_lsafRC_", returnCode);
      call symputx("_lsafMsg_", message);
    end;
  run;

%mend lsaf_updatestudydictionaries;
