%* lsaf_updatestdytlfstitlesfooters                                                       *;
%*                                                                                        *;
%* Updates the titles and footnotes of a study tlf using a SAS data set as input.         *;
%*                                                                                        *;
%* macrotype LSAF                                                                         *;
%* since 2.5                                                                              *;
%* exposure external                                                                      *;
%*                                                                                        *;
%* History                                                                                *;
%* 2021-03-25  initial coding                                                             *;
%* 2021-08-17  Remove header documentation                                                *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                         *;
%*                                                                                        *;

%macro lsaf_updatestdytlfstitlesfooters(
  lsaf_studyId=, 
  lsaf_basestandardname=,
  lsaf_modelId=,
  sas_dsname=,
  sas_results_dsname=
  ) /des='Updates the titles and footnotes of the TLFs in the input data set';

  %if "&sas_results_dsname" EQ "" %then %let sas_results_dsname=work.lsafupdatestdytlfstitlesfooters;


  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local logMsgPrefix
         logMsgPrimary
         macroName
         _tempUPDATE_tempDS_
		 haveDups
		 notesOption
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %***************************************************************************;
  %* Set local macro variables for processing.                               *;
  %***************************************************************************;
  %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
  %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );
  %let macroName=%sysfunc(lowcase(&SYSMACRONAME));

  %*************************************************************;
  %* Turn off the notes generated for all the pre-processing.  *;
  %* Save original option setting so it can be reset properly. *;
  %*************************************************************;
  %let notesOption=%sysfunc(getoption(notes));
  options nonotes;

  %***************************************************************************;
  %* Cleanup from any previous runs that may have left temp data set behind. *;
  %***************************************************************************;
  %let _tempUPDATE_tempDS_=%str(__SASMACRO_STDYTLFSTITLESFOOTS__);

  %if (%sysfunc(exist(&_tempUPDATE_tempDS_)) > 0) %then
  %do;
    %put NOTE: &macroName: Deleting temporary data set &_tempUPDATE_tempDS_ before processing.;
    proc datasets library=work nolist;
      delete &_tempUPDATE_tempDS_;
    quit; run;
  %end;

  %***************************************************************************;
  %* Validate the OUTPUT data set name. Do this now to prevent work being    *;
  %* done only to not be able to write out the results.                      *;
  %***************************************************************************;
  %lsaf_validate_sasdataset_name(callingMacro=&macroName,
        sas_dsn=&sas_results_dsname, dsType=Output
        );
 
  %if (&_lsafRC_ ne 0) %then
  %do;
    %GOTO PRINT_DS_MESSAGES;
  %end;

  %***************************************************************************;
  %* Validate the INPUT data set structure.                                  *;
  %* Sets global macro variable _lsafRC_=0 if validation successful          *;
  %***************************************************************************;
  %lsaf_validate_sasdataset(callingMacro=&macroName, sas_dsn=&sas_dsname,
        required_variables=%str(tlfIdentifier)
        );

  %if (&_lsafRC_ ne 0) %then
  %do;
    %GOTO PRINT_DS_MESSAGES;
  %end;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %***************************************************************************;
  %* If data set structures are valid, process input data set.               *;
  %***************************************************************************;
  %put NOTE: &macroName: Processing data set &sas_dsname.;

  %***************************************************************************;
  %* if input data set has only one variable...                              *;
  %* - it has to be tlfIdentifier (affirmed with above validation)           *;
  %* - so, there is nothing to transpose.                                    *;
  %***************************************************************************;
   %let dsid=%sysfunc(OPEN(&sas_dsname.,IN));
   %let VARCNT=%sysfunc(ATTRN(&dsid,NVARS));
   %let RC=%sysfunc(CLOSE(&dsid));
   %if (&VARCNT. EQ 1) %then 
   %do;
       %let _lsafMsg_ = %str(Input data set had no variables to process.);
       %GOTO PRINT_DS_MESSAGES;
   %end;

  %******************************************;
  %* Sort the data                          *;
  %******************************************;
  options nonotes;
  proc sort data=&sas_dsname. out=__lsaftempsorted__;
     by tlfIdentifier;
  run;

  %****************************************************;
  %* Confirm that the input data set does not         *;
  %* have multiple rows per tlfIdentifier, otherwise  *;
  %* the proc transpose will not work correctly.      *;
  %****************************************************;
  %let haveDups=0;
  data _null_;
    attrib tlfIds length=$1000;
    set __lsaftempsorted__ end=eof;
	by tlfIdentifier;
	retain tlfIds;
	if (first.tlfIdentifier AND not(last.tlfIdentifier)) then
	do;
	   tlfIds = kstrip(tlfIds) || " " || kstrip(tlfIdentifier);
	end;
	len=lengthn(kstrip(tlfIds));
	if (eof AND len>0) then
	do;
	   call symput("haveDups", "1");
	   call symput("_lsafMsg_", "TLF identifier must be unique in the data set. " ||
                   "The following identifiers appear more than once: " || kstrip(tlfIds));
	end;
  run;
  %if (&haveDups ne 0) %then
  %do;
    %GOTO PRINT_DS_MESSAGES;
  %end;

  %***************************************************************************;
  %* TRANSPOSE: Create vertical data set from horizontal one.                *;
  %***************************************************************************;
  proc transpose data=__lsaftempsorted__ out=&_tempUPDATE_tempDS_.;
     by tlfIdentifier;
     var _all_;
  run;

  %***************************************************************************;
  %* Clean up temporary sorted data set.                                     *;
  %***************************************************************************;
  proc datasets library=work nolist;
     delete __lsaftempsorted__;
  quit; run;

  option &notesOption.;

  %**********************************************;
  %* Process the vertical (transposed) records  *;
  %**********************************************;
  data &sas_results_dsname.(label="Created by &SYSMACRONAME."
                    keep=studyId baseStandardName modelId tlfIdentifier variable propertyName status
  );
    attrib returnCode    length=8
           message       length=$200
           updateCount   length=8
           rowCount      length=8
           msgPrefix     length=$64

           studyId            length=$64 
           baseStandardName   length=$15 
           modelId            length=$64
           tlfIdentifier      length=$256
           variable           length=$256
           propertyName       length=$256
           status             length=$64
    ;

    set &_tempUPDATE_tempDS_ end=eof;
    by tlfIdentifier;

    msgPrefix = "&logMsgPrefix.";

    %***********************************;
    %* Declare the java objects, once  *;
    %***********************************;
    if (_n_ = 1) then
    do;
      declare javaobj props("com/sas/lsaf/macro/clinical/study/tlf/SasQualifiedProperties");
      declare javaobj srv("com/sas/lsaf/macro/clinical/study/tlf/StudyTlfService") ;
      declare javaobj results("com/sas/lsaf/macro/clinical/SasClinicalPropertiesUpdateRecordsResult");

      put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * " %sysfunc(lowcase("&SYSMACRONAME"));
      put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * Input data set = &sas_dsname";
    end;

    if (first.tlfIdentifier) then
    do;
      props.callVoidMethod("clearList");
    end;

    %***********************************;
    %* Create a new properties list    *;
    %***********************************;
    props.callVoidMethod("addProperty", kstrip(_NAME_), kstrip(COL1), "");
    if (last.tlfIdentifier) then 
    do;
      GOTO DO_UPDATE;
    end;

    return;

    DO_UPDATE:

      %***********************************;
      %* Call the update                 *;
      %***********************************;
      put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * Processing tlf " tlfIdentifier;
      srv.callStaticVoidMethod("updateTlfTitlesAndFootnotes", "%bquote(&lsaf_studyId)", "%bquote(&lsaf_baseStandardName)", "%bquote(&lsaf_modelId)",
			kstrip(tlfIdentifier), props, results);

      %***********************************;
      %* Retrieve the results            *;
      %***********************************;
      results.callIntMethod("getReturnCode", returnCode);
      results.callStringMethod("getReturnMessage", message);

      %***********************************;
      %* Process the results             *;
      %***********************************;
      if (returnCode ne 0) then
      do;
        msgPrefix = "&logMsgPrimary.";
        put "ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: * Unable to process TLF " tlfIdentifier;
        GOTO ENDPROCESS;
      end;
      else do;
        updateCount+1;
      end;

      if (eof) then do;
         GOTO ENDPROCESS;
      end;

    RETURN;

    ENDPROCESS:
       results.callIntMethod("getSize", rowCount);
       studyId="&lsaf_studyId";
       baseStandardName="&lsaf_basestandardname"; 
       modelId="&lsaf_modelId";
       do row=0 to rowCount - 1;
           results.callStringMethod("getInputPropertyName", row, variable);
           results.callStringMethod("getSasPropertyName", row, propertyName);
           results.callStringMethod("getStatusMessage", row, status);
	   output;
	end;
		
        put msgPrefix " _lsafMsg_= " message ;
        put msgPrefix " _lsafRC_= " returnCode ;
		put " ";
    	put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * Successfully processed " updateCount "tlfs.";

        %***********************************;
        %* Set the macro variables         *;
        %***********************************;
        call symputx("_lsafRC_", returnCode);
        call symputx("_lsafMsg_", message);

        STOP;
  run;

  %GOTO CLEANUP_TEMP_DATASET;

  %PRINT_DS_MESSAGES:
    options &notesOption.;
    %put &logMsgPrimary _lsafMsg_= &_lsafMsg_ ;
    %put &logMsgPrefix &macroName ;
    %put &logMsgPrefix Input data set = &sas_dsname;
    %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;
    %put ;

  %************************************************************;
  %* Cleanup temporary data set if the operation did not fail *;
  %************************************************************;

  %CLEANUP_TEMP_DATASET:

    %let tempExists = %sysfunc(exist(&_tempUPDATE_tempDS_.));

    %if (&tempExists. eq 0) %then
    %do;
       %put &logMsgPrefix Temporary transposed data set work.&_tempUPDATE_tempDS_ was not created.;
       %GOTO FINISH_AND_LEAVE;
    %end;

    %if (&_lsafRC_ eq 0) %then
    %do;
       %put NOTE: &_LSAF_PRODUCT_NAME_ Macro: * %sysfunc(lowcase(&SYSMACRONAME)) is deleting temporary data set &_tempUPDATE_tempDS_..;
	   %put;
       options nonotes;
       proc datasets library=work nolist;
         delete &_tempUPDATE_tempDS_;
       quit; run;
	   options &notesOption.;
       %GOTO FINISH_AND_LEAVE;
    %end;

    %put WARNING: &_LSAF_PRODUCT_NAME_ Macro: * Temporary sorted data set work.&_tempUPDATE_tempDS_ is not being deleted.;

  %FINISH_AND_LEAVE:
    %put;
    options &notesOption.;

%mend lsaf_updatestdytlfstitlesfooters;
