%* lsaf_updateprocessflowproperties                                                              *;
%*                                                                                               *;
%* Updates the properties for the specified process flow.                                        *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 1.5                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2016-12-06  initial programming                                                               *;
%* 2019-02-12  update to 2.1                                                                     *;
%* 2021-08-12  Remove header documentation                                                       *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                                *;
%*                                                                                               *;

%macro lsaf_updateprocessflowproperties(
  lsaf_path=,
  lsaf_processflow=,
  sas_dsname=_lsaf_updatepf_donotupdate_
  ) /des='Updates a process flow in the specified SAS Life Science Analytics Framework context.';

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

   %global _lsafMsg_
           _lsafRC_ 
   ;

   %local logMsgPrefix
          logMsgPrimary
          macroName
		  notesOption
   ;

   %let _lsafRC_ =%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly. Unknown err%str(or).);
   %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
   %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );
   %let macroName=%sysfunc(lowcase(&SYSMACRONAME));

   %*************************************************************;
   %* Save original option setting so it can be reset properly. *;
   %*************************************************************;
   %let notesOption=%sysfunc(getoption(notes));

   %if (&sas_dsname=_lsaf_updatepf_donotupdate_) %then
   %do;
     %let _lsafMsg_= %str(No data has been provided for the update.);
     %GOTO PRINT_DSERROR_LOG_MSGS;
   %end;

   %if ( %CMPRES(&sas_dsname) NE %str(_lsaf_updatepf_donotupdate_) ) %then
   %do;
     %***************************************************************************;
     %* Validate the Process Flow Data data set if supplied.                    *;
     %* sets global macro variable _lsafRC_=0 if validation successful          *;
     %***************************************************************************;
	 options nonotes;
     %lsaf_validate_sasdataset(callingMacro=&macroName,
                               sas_dsn=&sas_dsname,
                               required_variables=%str(name value),
                               allow_empty_datasets=0);
     %if (&_lsafRC_ ne 0) %then
     %do;
       %let _lsafMsg_= sas_dsname: &_lsafMsg_;
       %GOTO PRINT_DSERROR_LOG_MSGS;
     %end;
   %end;

   option &notesOption.;

   %**********************************************;
   %* Process the records                        *;
   %**********************************************;
   data _null_;
     attrib returnCode      length=8
            message         length=$200
            logMsgPrefix    length=$70
            logMsgPrimary   length=$70
            logMsg          length=$300
            updatedName     length=$300
     ;

     set &sas_dsname end=eof;
     if (_n_ EQ 1) then
     do;
       %***********************************;
       %* Declare the java objects        *;
       %***********************************;
       declare javaobj srv("com/sas/lsaf/macro/workflow/processflow/ProcessFlowService");
       declare javaobj results("com/sas/lsaf/macro/common/result/Result");
       declare javaobj data("com/sas/lsaf/macro/common/SasProperties");
     end;

     data.callVoidMethod("addProperty", kstrip(name), kstrip(value));

     if (eof);

     %***********************************;
     %* Attempt the operation           *;
     %***********************************;
     srv.callStaticStringMethod("updateProperties", "%bquote(&lsaf_path)", "%bquote(&lsaf_processflow)",
                                data, results, updatedName);

     %***********************************;
     %* Retrieve the results            *;
     %***********************************;
     results.callIntMethod("getReturnCode", returnCode);
     results.callStringMethod("getReturnMessage", message);

     %***********************************;
     %* Process the results             *;
     %***********************************;
     logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
     if (returnCode ne 0) then
     do;
       logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
       logMsg=" Cannot update process flow properties for &lsaf_processflow in &lsaf_path.." ;
     end;
     else
     do;
       logMsgPrimary=logMsgPrefix;
       logMsg=" The process flow properties were updated for &lsaf_processflow in &lsaf_path.." ;
     end;

     %***********************************;
     %* Print messages in log           *;
     %***********************************;
     put;
     put logMsgPrimary " " logMsg;
     put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
     updatedName=kstrip(updatedName);
     if (returnCode=0 AND kstrip("&lsaf_processflow") NE updatedName) 
       then put logMsgPrefix " The process flow has been renamed to " updatedName ".";
     put logMsgPrefix " _lsafMsg_= " message ;
     put logMsgPrefix " _lsafRC_= " returnCode ;
     put;

     %***********************************;
     %* Set the macro variables         *;
     %***********************************;
     call symputx("_lsafRC_", returnCode);
     call symputx("_lsafMsg_", message);

   run;

   %return;

   %PRINT_DSERROR_LOG_MSGS:
      options &notesOption.;
      %put &logMsgPrimary _lsafMsg_= &_lsafMsg_ ;
      %put &logMsgPrefix  &macroName ;
      %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;
      %put;

%mend lsaf_updateprocessflowproperties;
