%* lsaf_updatepfsetupusercandidates                                                                 *;
%*                                                                                                  *;
%* Updates the candidates for process flow user task elements using a data set as input.            *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 1.5                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2017-01-16  original coding                                                                      *;
%* 2019-02-12  update to 2.1                                                                        *;
%* 2019-03-18  rename from lsaf_updatepfusercandidates to lsaf_updatepfsetupusercandidates          *;
%* 2021-08-12  Remove header documentation                                                          *;
%* 2021-10-21  principal refactor                                                                   *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                                   *;
%*                                                                                                  *;

%macro lsaf_updatepfsetupusercandidates(
  sas_dsname=
  ) /des='Update the candidates for process flow user task elements.';

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local logMsgPrefix
         logMsgPrimary
         macroName
		 notesOption
         tempDsMsg
         _tempUPDATE_tempDS_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let macroName=%sysfunc(lowcase(&SYSMACRONAME));

  %*************************************************************;
  %* Turn off the notes generated for all the pre-processing.  *;
  %* Save original option setting so it can be reset properly. *;
  %*************************************************************;
  %let notesOption=%sysfunc(getoption(notes));
  options nonotes;

  %***************************************************************************;
  %* Cleanup from any previous runs that may have left temp data set behind. *;
  %***************************************************************************;
  %let _tempUPDATE_tempDS_=%str(__SASMACRO_USERCANDIDATESTEMP__);

  %if (%sysfunc(exist(&_tempUPDATE_tempDS_)) > 0) %then
  %do;
    options &notesOption.;
    %put NOTE: &macroName: Deleting temporary data set &_tempUPDATE_tempDS_ before processing.;
	option nonotes;
    proc datasets library=work nolist;
      delete &_tempUPDATE_tempDS_;
    quit; run;
  %end;

  %***************************************************************************;
  %* Set local macro variables for processing.                               *;
  %*                                                                         *;
  %* setting a new macro variable for operation success because there are    *;
  %* operations in updateACL that could set _lsafRC_ to something other than *;
  %* 0 but still not be a failure.                                           *;
  %***************************************************************************;
  %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
  %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );
  %let tempDsMsg=%str(Temporary sorted data set work.&_tempUPDATE_tempDS_ will not be created.);

  %***************************************************************************;
  %* Validate the input data set                                             *;
  %* sets global macro variable _lsafRC_=0 if validation successful          *;
  %***************************************************************************;
  %lsaf_validate_sasdataset(callingMacro=&macroName,
        sas_dsn=&sas_dsname,
        required_variables=%str(processFlowPath processFlowName elementID type principalId grpSrcCtxt)
        );
  options &notesOption.;
  %if (&_lsafRC_ ne 0) %then
  %do;
    %put &logMsgPrimary _lsafMsg_= &_lsafMsg_ ;
    %put &logMsgPrefix &macroName ;
    %put &logMsgPrefix Input data set = &sas_dsname;
    %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;
    %put &logMsgPrefix &tempDsMsg ;
    %put;
    %return;
  %end;

  %***************************************************************************;
  %* If data set is valid, process it.                                       *;
  %***************************************************************************;
  %put NOTE: &macroName: Processing data set &sas_dsname;

  %************************;
  %* Sort the data        *;
  %************************;
  options nonotes;
  proc sort data=&sas_dsname. out=&_tempUPDATE_tempDS_.;
    by processFlowPath processFlowName elementID;
  run;

  options &notesOption.;

  %**********************************************;
  %* Process the records                        *;
  %**********************************************;
  data _null_;
    attrib returnCode    length=8
           message       length=$200
           updateCount   length=8
    ;

    set &_tempUPDATE_tempDS_ end=eof;
    by processFlowPath processFlowName elementID;

    %***********************************;
    %* Declare the java objects, once  *;
    %***********************************;
    if (_n_ = 1) then
    do;
      declare javaobj candidates("com/sas/lsaf/macro/security/SasPrincipalDescriptors");
      declare javaobj srv("com/sas/lsaf/macro/workflow/setup/ProcessFlowSetupService");
      declare javaobj results("com/sas/lsaf/macro/common/result/Result");

      put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * " %sysfunc(lowcase("&SYSMACRONAME"));
      put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * Input data set = &sas_dsname";
    end;

    if (first.elementID) then
    do;
      candidates.callVoidMethod("clearList");
      %***********************************************;
      %* Do not want to add a blank candidate if the *;
      %* user is trying to clear out the list.       *;
      %***********************************************;
      if (last.elementID AND kstrip(type)="" AND kstrip(principalId)="") 
        then GOTO DO_UPDATE;
    end;

    %***********************************;
    %* Create a new candidatess list   *;
    %***********************************;
    candidates.callVoidMethod("addValue", kstrip(type), kstrip(principalId), kstrip(grpSrcCtxt));

    if (last.elementID) then GOTO DO_UPDATE;

    RETURN;

    DO_UPDATE:
      %***********************************;
      %* Call the update                 *;
      %***********************************;
      put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * Processing element " elementID " in process flow " processFlowName " in " processFlowPath;
      srv.callStaticVoidMethod("updateElementPrincipals", kstrip(processflowpath), kstrip(processflowname), kstrip(elementID),
           "%bquote(USER_TASK)", candidates, results);

      %***********************************;
      %* Retrieve the results            *;
      %***********************************;
      results.callIntMethod("getReturnCode", returnCode);
      results.callStringMethod("getReturnMessage", message);

      %***********************************;
      %* Process the results             *;
      %***********************************;
      if (returnCode ne 0) then
      do;
        put "ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: * Unable to process " elementID;
        put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * _lsafMsg_ = " message;
        put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * _lsafRC_ = " returnCode;
        put;
        GOTO ENDPROCESS;
      end;
      else updateCount+1;

      if (eof) then GOTO ENDPROCESS;

    RETURN;

    ENDPROCESS:
    put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * Successfully processed " updateCount "user task elements";
    put " ";

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);

    STOP;

  run;

  %************************************************************;
  %* Cleanup temporary data set if the operation did not fail *;
  %************************************************************;
  %if (&_lsafRC_ eq 0) %then
  %do;
    %if (%sysfunc(exist(&_tempUPDATE_tempDS_))>0) %then
    %do;
      %put NOTE: &_LSAF_PRODUCT_NAME_ Macro: * %sysfunc(lowcase(&SYSMACRONAME)) is deleting temporary data set &_tempUPDATE_tempDS_;
      options nonotes;
      proc datasets library=work nolist;
        delete &_tempUPDATE_tempDS_;
        quit; run;
    %end;
  %end;
  %else %put WARNING: &_LSAF_PRODUCT_NAME_ Macro: * Temporary sorted data set work.&_tempUPDATE_tempDS_ is not being deleted.;

  options &notesOption;

%mend lsaf_updatepfsetupusercandidates;
