%* lsaf_updatepfsetuptimers                                                                      *;
%*                                                                                               *;
%* Updates the values of process flow timer elements using a data set as input. Each row of the  *;
%* data set represents a timer element.                                                          *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2017-01-03  original code                                                                     *;
%* 2019-02-12  update to 2.1                                                                     *;
%* 2019-03-18  rename from lsaf_updatepftimerelements to lsaf_updatepfsetuptimers                *;
%* 2021-08-12  Remove header documentation                                                       *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                                *;
%*                                                                                               *;

%macro lsaf_updatepfsetuptimers(
  sas_dsname=
  ) /des='Updates timer elements for process flows in the SAS Life Science Analytics Framework.';


  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %***************************************************************************;
  %* Set local macro variables for processing.                               *;
  %***************************************************************************;
  %local logMsgPrefix
         logMsgPrimary
         macroName
		 notesOption
         ;

  %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
  %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );
  %let macroName=%sysfunc(lowcase(&SYSMACRONAME));

  %*************************************************************;
  %* Turn off the notes generated for all the pre-processing.  *;
  %* Save original option setting so it can be reset properly. *;
  %*************************************************************;
  %let notesOption=%sysfunc(getoption(notes));
  options nonotes;

  %***************************************************************************;
  %* Validate the input data set                                             *;
  %* sets global macro variable _lsafRC_=0 if validation successful          *;
  %***************************************************************************;
  %lsaf_validate_sasdataset(callingMacro=&macroName,
        sas_dsn=&sas_dsname,
        required_variables=%str(processFlowPath processFlowName elementID value years months days hours minutes seconds),
        required_numeric_variables=%str(years months days hours minutes seconds),
        allow_empty_datasets=0);

  options &notesOption.;

  %if (&_lsafRC_ ne 0) %then
  %do;
    %put &logMsgPrimary _lsafMsg_= &_lsafMsg_ ;
    %put &logMsgPrefix &macroName ;
    %put &logMsgPrefix Input data set = &sas_dsname;
    %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;
    %put;
    %return;
  %end;

  %***************************************************************************;
  %* If data set is valid, process it.                                       *;
  %***************************************************************************;
  %put NOTE: &macroName: Processing data set &sas_dsname;

  %**********************************************;
  %* Process the records                        *;
  %**********************************************;
  data _null_;
    attrib returnCode       length=8
           message          length=$200
           updatedCnt       length=8
    ;

    set &sas_dsname end=eof;

    %***********************************;
    %* Declare the java objects, once  *;
    %***********************************;
    if (_n_ = 1) then
    do;
      declare javaobj srv("com/sas/lsaf/macro/workflow/setup/ProcessFlowSetupService");
      declare javaobj results("com/sas/lsaf/macro/common/result/Result");
      declare javaobj timer("com/sas/lsaf/macro/workflow/setup/SasProcessFlowTimerElement");

      put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * " %sysfunc(lowcase("&SYSMACRONAME"));
      put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * Input data set = &sas_dsname";
    end;

    %***********************************;
    %* Set timer values                *;
    %***********************************;
    timer.callVoidMethod("setTimerValues", kstrip(value), years, months, days, hours, minutes, seconds);

    %***********************************;
    %* Call the update                 *;
    %***********************************;
    put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * Processing " elementID "for " processFlowName "in " processFlowPath;
    srv.callStaticVoidMethod("updateTimerElement", kstrip(processFlowPath), kstrip(processFlowName), kstrip(elementID), timer, results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    if (returnCode ne 0) then
    do;
      put "ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: * Unable to process " elementID "for " processFlowName "in " processFlowPath ".";
      put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * _lsafMsg_ = " message;
      put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * _lsafRC_ = " returnCode;
      put;
      GOTO ENDPROCESS;
    end;
    else updatedCnt+1;

    if (eof) then GOTO ENDPROCESS;

    return;

    ENDPROCESS:
    put "NOTE: &_LSAF_PRODUCT_NAME_ Macro: * Successfully processed " updatedCnt "timer elements";
    put " ";

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    STOP;

  run;

%mend lsaf_updatepfsetuptimers;
