%* lsaf_updateacls                                                                                 *;
%*                                                                                                 *;
%* Sets the permissions on SAS Life Science Analytics Framework items using a SAS data set as      *;
%* input.                                                                                          *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 1.2                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2013-08-30  updates for standardization and coding conventions                                  *;
%* 2016-08-26  rebrand as LSAF                                                                     *;
%* 2016-02-09  updates for path sorting, notes to log, stop after first failure.                   *;
%* 2019-02-05  update to 2.1                                                                       *;
%* 2020-07-13  update for refactor of java service hierarchy.                                      *;
%* 2021-08-05  Remove header documentation                                                         *;
%* 2021-09-23  DE16976 Updated sort to remove excessive processing.                                *;
%* 2023-03-21  DE17654 - SAS Notes option updates                                                  *;
%*                                                                                                 *;

%macro lsaf_updateacls(
  sas_dsname=
  ) /des='Set permissions on SAS Life Science Analytics Framework items.';

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;
  %local notesOption
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %*************************************************************;
  %* Turn off the notes generated for all the pre-processing.  *;
  %* Save original option setting so it can be reset properly. *;
  %*************************************************************;
  %let notesOption=%sysfunc(getoption(notes));

  %***************************************************************************;
  %* Cleanup from any previous runs that may have left temp data set behind. *;
  %***************************************************************************;
  %let _tempACLUPDATE_tempDS_=%str(__SASMacro_acltemp__);

  %if (%sysfunc(exist(&_tempACLUPDATE_tempDS_)) > 0) %then
  %do;
    %put NOTE: lsaf_updateacls: Deleting temporary data set &_tempACLUPDATE_tempDS_ before processing.;
    options nonotes;
    proc datasets library=work nolist;
      delete &_tempACLUPDATE_tempDS_;
    quit; run;
    options &notesOption.;
  %end;

  %***************************************************************************;
  %* Set local macro variables for processing.                               *;
  %*                                                                         *;
  %* setting a new macro variable for operation success because there are    *;
  %* operations in updateACL that could set _lsafRC_ to something other than *;
  %* 0 but still not be a failure.                                           *;
  %***************************************************************************;
  %local logMsgPrefix logMsgPrimary tempDsMsg;
  %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
  %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );
  %let tempDsMsg=%str(Temporary sorted data set work.&_tempACLUPDATE_tempDS_ will not be created.);

  %***************************************************************************;
  %* Validate the input data set                                             *;
  %* sets global macro variable _lsafRC_=0 if validation successful          *;
  %***************************************************************************;
  %let macroName=%sysfunc(lowcase(&SYSMACRONAME));
  %lsaf_validate_sasdataset(callingMacro=&macroName,
        sas_dsn=&sas_dsname,
        required_variables=%str(path aclType aclPrincipalType aclPrincipalName grpSrcCtxt
                    adminPermission readPermission writePropPerm writeContentPerm deletePermission),
        required_numeric_variables=%str(adminPermission readPermission writePropPerm writeContentPerm deletePermission)
        );
  options &notesOption.;
  %if (&_lsafRC_ ne 0) %then
  %do;
    %put &logMsgPrimary &macroName - &tempDsMsg;
    %put &logMsgPrefix Input data set = &sas_dsname;
    %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;
    %put &logMsgPrefix _lsafMsg_= &_lsafMsg_ ;
    %put;
    %return;
  %end;

  %***************************************************************************;
  %* If data set is valid, process it.                                       *;
  %***************************************************************************;
  %put NOTE: &macroName: Processing data set &sas_dsname;

  %************************;
  %* Sort the data        *;
  %************************;
  option nonotes;
  proc sort data=&sas_dsname. out=&_tempACLUPDATE_tempDS_
		sortseq=linguistic(strength=primary);
    by path aclType;
  run;
  options &notesOption.;

  %**********************************************;
  %* Process the records                        *;
  %**********************************************;
  data _null_;
    attrib returnCode               length=8;
    attrib message                  length=$200;
    attrib emptyAceCount            length=8;
    attrib aclUpdateCount           length=8;

    set &_tempACLUPDATE_tempDS_ end=eof;
    by path aclType;

    %***********************************;
    %* Declare the java items, once    *;
    %***********************************;
    if (_n_ = 1) then
    do;
      declare javaobj acl("com/sas/lsaf/macro/security/acl/SasAcl");
      declare javaobj srv("com/sas/lsaf/macro/security/acl/AclService");
      declare javaobj results("com/sas/lsaf/macro/security/acl/SasUpdateAclResult");

      put "&logMsgPrefix * " %sysfunc(lowcase("&SYSMACRONAME"));
      put "&logMsgPrefix * Input data set = &sas_dsname";
    end;

    if (first.aclType) then acl.callVoidMethod("resetAcl", kstrip(path), kstrip(aclType));

    %***********************************;
    %* Create a new ace                *;
    %***********************************;
    acl.callVoidMethod("addAce", kstrip(aclPrincipalType), kstrip(aclPrincipalName), kstrip(grpSrcCtxt),
                        adminPermission, readPermission, writePropPerm, writeContentPerm, deletePermission);

    if (last.aclType) then
    do;

      %***********************************;
      %* Call the update                 *;
      %***********************************;
      put "&logMsgPrefix * Processing " aclType "ACL for path " path;
      srv.callStaticVoidMethod("updateAcl", acl, results);

      %***********************************;
      %* Retrieve the results            *;
      %***********************************;
      results.callIntMethod("getReturnCode", returnCode);
      results.callStringMethod("getReturnMessage", message);

      %***********************************;
      %* Process the results             *;
      %***********************************;
      if (returnCode ne 0) then
      do;
        put "&logMsgPrimary * Unable to process " aclType "ACL for path " path;
        put "&logMsgPrefix * _lsafMsg_ = " message;
        put "&logMsgPrefix * _lsafRC_ = " returnCode;
        put;
        GOTO ENDPROCESS;
      end;
    end;

    if (eof) then GOTO ENDPROCESS;

    return;

    ENDPROCESS:
    results.callIntMethod("getAclUpdateCount", aclUpdateCount);
    results.callIntMethod("getBlankAceCount", emptyAceCount);
    if (emptyAceCount gt 0) 
      then put "WARNING: &_LSAF_PRODUCT_NAME_ Macro: * " emptyAceCount "record(s) had no permissions "
               "explicitly set for the ACL being processed.  The ACE for those records was not "
               "added or was removed from the ACL.";

    put "&logMsgPrefix * Successfully processed " aclUpdateCount "ACLs";
    put " ";

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    STOP;
  run;

  %************************************************************;
  %* Cleanup temporary data set if the operation did not fail *;
  %************************************************************;
  %if (&_lsafRC_ eq 0) %then
  %do;
      %if (%sysfunc(exist(&_tempACLUPDATE_tempDS_))>0) %then
      %do;
        %put &logMsgPrefix * %sysfunc(lowcase(&SYSMACRONAME)) is deleting temporary data set &_tempACLUPDATE_tempDS_;
         proc datasets library=work nolist;
            delete &_tempACLUPDATE_tempDS_;
         quit; run;
      %end;
  %end;
  %else
  %do;
      %put WARNING: &_LSAF_PRODUCT_NAME_ Macro: * Temporary sorted data set work.&_tempACLUPDATE_tempDS_ is not being deleted.;
  %end;

  options &notesOption.;

%mend lsaf_updateacls;
