%* lsaf_search                                                                                   *;
%*                                                                                               *;
%* Searches for an item in the repository. Although all parameters are optional, at least one    *;
%* parameter value must be specified to start a search.                                          *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 2.2                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2019-05-10  Initial development                                                               *;
%* 2019-11-01  update to handle a blank SAS output file.                                         *;
%* 2021-08-13  Remove header documentation                                                       *;
%* 2021-08-25  DE16765                                                                           *;
%* 2021-08-31  DE16792                                                                           *;
%* 2021-10-12  US11739                                                                           *;
%*                                                                                               *;

%macro lsaf_search(
  lsaf_location=,
  lsaf_name=,
  lsaf_typeid=,
  lsaf_owner=,
  lsaf_lastmodifiedby=,
  lsaf_createdby=,
  lsaf_fromcreated=,
  lsaf_tocreated=,
  lsaf_fromlastmodified=,
  lsaf_tolastmodified=,
  lsaf_content=,
  lsaf_copytoworkspacestatus=,
  sas_dsname=work.lsafSearch,
  lsaf_printLimitReachedWarning=1
  ) /des='Search for an item in the repository.';

  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafSearch;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
          _lsaf_PrintMessage_
          _lsafSearchLimitReached_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname(label="Created by &SYSMACRONAME"
                   keep=path name itemType isContainer description version isVersioned isCheckedOut isLocked signatureStatus 
                        size formattedsize createdBy created dateCreated lastModifiedBy lastModified dateLastModified
                        propertiesLastModifiedBy propertiesLastModified datePropertiesLastModified state);
    attrib returnCode                length=8
           message                   length=$200
           logMsgPrefix              length=$70
           logMsg                    length=$300
           logMsgPrimary             length=$200
           rowCount                  length=8

           path                      length=$2048    label="Item Path"
           name                      length=$256     label="Item Name"
           itemType                  length=$32      label="Item Type"
           isContainer               length=8        label="Is Item A Container?"

           description               length=$512     label="Description"
           version                   length=$32      label="Version Number"
           isVersioned               length=8        label="Is File Versioned"
           isCheckedOut              length=8        label="Is File Checked out"
           isLocked                  length=8        label="Is File Locked"
           signatureStatus           length=$32      label="Is File Signed"
           size                      length=8        label="Size In Bytes"
           formattedsize             length=$20      label="Size (Formatted)"

           createdBy                 length=$64      label="Userid that created the item"
           created                   length=$64      label="Item Creation Date"
           dateCreated                               label="Date the item was created (SAS DateTime Format)"

           lastModifiedBy            length=$64      label="Userid that last modified the item content"
           lastModified              length=$64      label="Last Content Modification Date"
           dateLastModified                          label="Date the item's content was last modified (SAS DateTime Format)"

           propertiesLastModifiedBy  length=$64      label="Userid that last modified the item properties"
           propertiesLastModified    length=$64      label="Last Properties Modification Date"
           datePropertiesLastModified                label="Date the item's properties were last modified (SAS DateTime Format)"

           state                     length=$64      label="State of a context item"

           createdTempSAS            length=$64      label="TEMPORARY - WILL BE DROPPED"
           lastModifiedTempSAS       length=$64      label="TEMPORARY - WILL BE DROPPED"
           propertiesLastModifiedTempSAS  length=$64 label="TEMPORARY - WILL BE DROPPED"
    ;

    format dateCreated
           dateLastModified
           datePropertiesLastModified datetime.
    ;

    rowCount=0;


    %***********************************;
    %* Declare the java items        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/content/repository/RepositoryService");
    declare javaobj results("com/sas/lsaf/macro/content/repository/SasRepositoryItemsSearchResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("search", "%bquote(&lsaf_location)", "%bquote(&lsaf_name)", "%bquote(&lsaf_typeid)", "%bquote(&lsaf_owner)", "%bquote(&lsaf_lastmodifiedby)", 
                 "%bquote(&lsaf_createdby)", "%bquote(&lsaf_fromcreated)", "%bquote(&lsaf_tocreated)", "%bquote(&lsaf_fromlastmodified)", "%bquote(&lsaf_tolastmodified)", 
                 "%bquote(&lsaf_content)", "%bquote(&lsaf_copytoworkspacestatus)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);
    results.callBooleanMethod("isLimitReached", limitReached);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot find search item(s)." ;
    end;
    else
    do;
      results.callIntMethod("getSize", rowCount);
      if limitReached then
      do;
        %lsaf_printwarningmessage(lsaf_printmessage=&lsaf_printlimitreachedwarning);
        logMsgPrimary="&_lsaf_PrintMessage_";
        logMsg=" Items were found matching the search criteria.";
      end;
      else 
      do;
        logMsgPrimary=logMsgPrefix;
        if (rowCount=0) 
          then logMsg = " No items were found matching the search criteria.";
          else logMsg = " Items were found matching the search criteria.";
      end;
      do row=0 to rowCount - 1;
        results.callStringMethod("getPath", row, path);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getDescription", row, description);
        results.callStringMethod("getVersion", row, version);
        results.callIntMethod("isContainer", row, isContainer);
        results.callIntMethod("isVersioned", row, isVersioned);
        results.callIntMethod("isCheckedOut", row, isCheckedOut);
        results.callIntMethod("isLocked", row, isLocked);
        results.callStringMethod("getSignatureStatus", row, signatureStatus);
        results.callLongMethod("getFileSize", row, size);
        if size < 0 then size=.;
        results.callStringMethod("getFormattedFileSize", row, formattedsize); 
        results.callStringMethod("getItemType", row, itemType);
        results.callStringMethod("getCreated", row, created);
        results.callStringMethod("getCreatedBy", row, createdBy);
        results.callStringMethod("getLastModified", row, lastModified);
        results.callStringMethod("getLastModifiedBy", row, lastModifiedBy);
        results.callStringMethod("getPropertiesLastModified", row, propertiesLastModified);
        results.callStringMethod("getPropertiesLastModifiedBy", row, propertiesLastModifiedBy);
        results.callStringMethod("getState", row, state);

        results.callStringMethod("getCreatedSasFormat", row, createdTempSas);
        results.callStringMethod("getLastModifiedSasFormat", row, lastModifiedTempSas);
        results.callStringMethod("getPropertiesLastModifiedSasFormat", row, propertiesLastModifiedTempSAS);

        dateCreated=input(createdTempSAS, datetime.);
        dateLastModified=input(lastModifiedTempSas, datetime.);
        datePropertiesLastModified=input(propertiesLastModifiedTempSAS, datetime.);

        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put logMsgPrefix " _lsafSearchLimitReached_= " limitReached;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);
    call symputx("_lsafSearchLimitReached_", limitReached);

  run;
  
  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by path;
    run;
    options &notesOption;
  %end;

%mend lsaf_search;
