%* lsaf_query                                                                                               *;
%*                                                                                                          *;
%* Performs a query and returns an output file to a specified location                                      *;
%*                                                                                                          *;
%* macrotype LSAF                                                                                           *;
%* since 2.3                                                                                                *;
%* exposure internal                                                                                        *;
%*                                                                                                          *;
%* History                                                                                                  *;
%* 2020-04-01  initial coding                                                                               *;
%* 2020-06-09  expand parameter list to include path and scope                                              *;
%* 2021-02-18  add parameter allowing for a missing query dataset (default=0, No)                           *;
%* 2021-08-12  Remove header documentation                                                                  *;
%* 2021-09-20  Remove parameter defaults (DE16775)                                                          *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                                           *;
%*                                                                                                          *;

%macro lsaf_query(
  lsaf_callingMacro=,
  lsaf_path=,
  lsaf_scope=,
  lsaf_queryMethod=,
  lsaf_returnPathVar=,
  lsaf_querydataset=,
  lsaf_validateddataset=,
  lsaf_exportlocation=,
  lsaf_exportpath=,
  lsaf_overwrite=,
  lsaf_enableversioning=,
  lsaf_versiontype=,
  lsaf_customversion=,
  lsaf_comment=,
  lsaf_allowmissingqueryds=0
  ) /des='Performs a query and returns an output file to a specified location in the SAS Life Science Analytics Framework.';

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
          &lsaf_returnPathVar.
  ;
  %local nonotes;

  %let _lsafRowCount_=0;
  %let _lsafRC_=0;
  %let _lsafMsg_=;
  %let _skipQueryDs=0;
  %let notesOption=%sysfunc(getoption(notes));
  
  %if &lsaf_allowmissingqueryds eq 1 AND "&lsaf_querydataset" EQ "" %then 
  %do;
    %let _skipQueryDs = 1;
    %goto START_PROCESS;
  %end;
  
  %*************************************************************;
  %* Turn off the notes generated for all the pre-processing.  *;
  %*************************************************************;
  options nonotes;

  %*******************************************************************;
  %* Validate the input data set                                     *;
  %* sets global macro variable _lsafRC_=0 if validation successful  *;
  %*******************************************************************;
  %let macroName=%sysfunc(lowcase(&SYSMACRONAME));
  %lsaf_validate_sasdataset(callingMacro=&lsaf_callingMacro, sas_dsn=&lsaf_querydataset, 
                          required_variables=%str(recordType columnClass columnName value comparator),
                          required_numeric_variables=%str(isCaseSensitive)
                          );
  
  options &notesOption.;
  
  %*******************************************************************;
  %* Reset _lsafRC_ from -999 to -1 for empty datasets or miss vars  *;
  %*******************************************************************;

  %if %sysfunc(find(%upcase(&_lsafMsg_),NO OBSERVATIONS)) %then %let _lsafRC_=-1;
  %if %sysfunc(find(%upcase(&_lsafMsg_),MISSING REQUIRED)) %then %let _lsafRC_=-1;

  %if (&_lsafRC_ ne 0) %then
  %do;
    %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
    %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );
    %put &logMsgPrimary &_lsafMsg_ ;
    %put &logMsgPrefix &lsaf_callingMacro ;
    %put &logMsgPrefix The query was not performed.;
    %put &logMsgPrefix Input data set = &lsaf_querydataset;
    %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;
    %put;
    %return;
  %end;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %START_PROCESS:

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &lsaf_validateddataset(keep=recordType columnClass columnName value comparator isCaseSensitive columnType validationNote);
    attrib returnCode       length=8
           message          length=$200
           logMsgPrefix     length=$70
           logMsgPrimary    length=$70
           logMsg           length=$300
           exportPath       length=$2048
           
           recordType       length=$32    label="Record Type"
           columnClass      length=$32    label="Column Class"
           columnName       length=$32    label="Column Name"
           value            length=$2048  label="Value Type"
           comparator       length=$32    label="Comparator"
           isCaseSensitive  length=8      label="Is Case Sensitive"
           columnType       length=$25    label="Column Type"
           validationNote   length=$200   label="Validation Note"

           queryConstraint  length=$5000
    ;
    
    %if &_skipQueryDs EQ 0 %then
    %do;
      set &lsaf_querydataset end=eof;
    %end;
    %else
    %do;
      _n_=1;
      eof=1;
    %end;

    %***********************************;
    %* Declare the java objects, once  *;
    %***********************************;
    if (_n_ = 1) then
    do;
      declare javaobj srv("com/sas/lsaf/macro/query/QueryService") ;
      declare javaobj results("com/sas/lsaf/macro/query/SasQueryResult");
      declare javaobj inputData("com/sas/lsaf/macro/query/SasQueryRecords");        
      declare javaobj exportData("com/sas/lsaf/macro/content/SasFileCreateParameters"); 
      exportData.callVoidMethod("updateData", "%bquote(&lsaf_exportLocation)", "%bquote(&lsaf_exportpath)", "%bquote(&lsaf_overwrite)", 
                                "%bquote(&lsaf_enableversioning)", "%bquote(&lsaf_versiontype)", "%bquote(&lsaf_customversion)", 
                                "%bquote(&lsaf_comment)");
    end;
    
    %if &_skipQueryDs EQ 0 %then
    %do;
      inputData.callVoidMethod("addQueryRecord", kstrip(recordType), kstrip(columnClass), kstrip(columnName), kstrip(value), kstrip(comparator),
                              isCaseSensitive);
    %end;

    if eof;

    rowCount=0;

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    %if ("%lowcase(&macName)" EQ "lsaf_querycontextmembership") OR ("%lowcase(&macName)" EQ "lsaf_querydistcontextmembership") %then
    %do;
      srv.callStaticStringMethod("&lsaf_queryMethod.", "&lsaf_path.", "&lsaf_scope.", inputData, exportData, results, exportPath);
    %end;
    %else
    %do;
      srv.callStaticStringMethod("&lsaf_queryMethod.", inputData, exportData, results, exportPath);
    %end;
    
    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      if returnCode=-200 then 
        logMsg=" Could not perform the query, see the validated data set: &lsaf_validateddataset.." ;
      else
        logMsg=" The query was not performed due to an error." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Returned the export file path and the validated data set: &lsaf_validateddataset..";
    end;
    results.callIntMethod("getSize", rowCount);
    do row=0 to rowCount - 1;
      results.callStringMethod("getRecordType", row, recordType);
      results.callStringMethod("getColumnClass", row, columnClass);
      results.callStringMethod("getColumnName", row, columnName);
      results.callStringMethod("getValue", row, value);
      results.callStringMethod("getComparator", row, comparator);
      results.callIntMethod("getIsCaseSensitive", row, isCaseSensitive);
      results.callStringMethod("getColumnType", row, columnType);
      results.callStringMethod("getNote", row, validationNote);
      output;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " &lsaf_callingMacro.";
    if (returnCode ne 0) 
      then put logMsgPrefix " The query was not performed.";
      else put logMsgPrefix " The query was performed and results were exported to " exportPath;
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put logMsgPrefix " &lsaf_returnPathVar= " exportPath;

    %*******************************;
    %* Output query to the SAS log *;
    %*******************************; 
    results.callIntMethod("getSelectsSize", rowCount);
    if (rowCount > 0) then 
    do;
      put logMsgPrefix " Query= ";
      put "SELECT: [ " @;
      do row=0 to rowCount-1;
        results.callStringMethod("getSelectColumnName", row, columnName);
        put columnName @;
        if (row < rowCount-1) then put +(-1) ", " @;
      end;
      put "]";
      results.callIntMethod("getConstraintsSize", rowCount);
      results.callStringMethod("getQueryLogicalOperator", comparator);
      put "CONSTRAINTS: [";
      do row=0 to rowCount-1;
        results.callStringMethod("getConstraint", row, queryConstraint);
        queryConstraint = strip(queryConstraint);
        put "    " queryConstraint;
        if (row<rowCount-1) then put "  " comparator;
      end;
      put "]";
      results.callIntMethod("getOrdersSize", rowCount);
      if (rowCount > 0) then 
      do;
        isAscending = -1;
        put "ORDER: [ " @;
        if (rowCount>1) then put;
        do row=0 to rowCount-1;
          if (rowCount>1) then put "   [ " @;
          results.callStringMethod("getOrderColumnName", row, columnName);
          results.callBooleanMethod("getOrderIsAscending", row, isAscending);
          results.callBooleanMethod("getOrderIsCaseSensitive", row, isCaseSensitive);
          put columnName @;
          if (isAscending eq 1) 
            then put " ascending " @;
            else put " descending " @;
          if (isCaseSensitive) then put " (case-sensitive) " @;
          if (rowCount>1) then put "] ";
        end;
        put "]";
      end; 
    end;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    put;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("&lsaf_returnPathVar.", kstrip(exportPath));
  run;

%mend lsaf_query;
