%* lsaf_process_jobdatasets                                                                          *;
%*                                                                                                   *;
%* Validates the data sets that are specified by the user to create or update jobs. The data sets    *;
%* are combined into one central data set for processing.                                            *;
%*                                                                                                   *;
%* Syntax:                                                                                           *;
%* %LSAF_PROCESS_JOBDATASETS(CALLINGMACRO=calling-macro <, sas_dsname_info,                          *;
%*     sas_dsname_programs, sas_dsname_parameters, sas_dsname_setup, sas_dsname_checkin> );          *;
%*                                                                                                   *;
%* macvar _lsafRC_  the return code.                                                                 *;
%* macvar _lsafMsg_ the return message.                                                              *;
%*                                                                                                   *;
%* macrotype LSAF                                                                                    *;
%* since 1.5                                                                                         *;
%* exposure internal                                                                                 *;
%*                                                                                                   *;
%* History                                                                                           *;
%* 2016-09-26  initial coding                                                                        *;
%* 2016-11-07  modifications for update jobs                                                         *;
%* 2019-02-27  modifications for 2.1                                                                 *;
%* 2020-08-10  modifications for LSAF 5.3 compatibility.                                             *;
%* 2021-08-18  Update header documentation                                                           *;
%*                                                                                                   *;

%macro lsaf_process_jobdatasets(
   callingMacro=,
   sas_dsname_info=,
   sas_dsname_programs=,
   sas_dsname_parameters=,
   sas_dsname_setup=,
   sas_dsname_checkin=
   ) / des='Validates SAS data sets for use in creating and updating jobs.';

   %let sas_dsname_info=%CMPRES(&sas_dsname_info);
   %let sas_dsname_programs=%CMPRES(&sas_dsname_programs);
   %let sas_dsname_parameters=%CMPRES(&sas_dsname_parameters);
   %let sas_dsname_setup=%CMPRES(&sas_dsname_setup);
   %let sas_dsname_checkin=%CMPRES(&sas_dsname_checkin);

   %let notprovidedstring=%str(_&callingMacro._ds_notprovided_);
   %if ( %index(%upcase(&callingMacro), UPDATE) NE 0) %then
      %let updating=1;
   %else
      %let updating=0;

   %****************************************;
   %* Initialize macro variables           *;
   %****************************************;
   %let _lsafRC_=%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

   %global _lsafMASTERJobDATA_tempDS_
           source_info
           source_programs
           source_parms
           source_setup
           source_checkin
   ;

   %let _lsafMASTERJobDATA_tempDS_=%str(_null_);
   %let _lsaf_tempDS_=%str(_null_);
   %let _tempSETUP_tempDS_=%str(_null_);
   %let _tempCHECKIN_tempDS_=%str(_null_);
   %let _tempPARAMETERS_tempDS_=%str(_null_);

   %if (&sas_dsname_info=&notprovidedstring AND &sas_dsname_programs=&notprovidedstring AND
        &sas_dsname_parameters=&notprovidedstring AND &sas_dsname_setup=&notprovidedstring AND
        &sas_dsname_checkin=&notprovidedstring) %then
      %do;
         %if (&updating) %then
            %let _lsafMsg_= %str(No data has been provided for the update.);
         %else
            %let _lsafRC_=0;
         %RETURN;
      %end;

   %let source_info=INFO;
   %let source_programs=PROGRAMS;
   %let source_parms=PARMS;
   %let source_setup=SETUP;
   %let source_checkin=CHECKIN;


   %let _lsafMASTERJobDATA_tempDS_=%str(__LSAF_SASMacro_createjobtemp__);
   %let _lsaf_tempDS_=%str(__LSAF_temp__);

   %*************************************************************;
   %* Turn off the notes generated for all the pre-processing.  *;
   %* Save original option setting so it can be reset properly. *;
   %*************************************************************;

   options nonotes;

   %put NOTE: &macroName: Processing job data sets.;

   %***************************************************************************;
   %* Cleanup from any previous runs that may have left temp data set behind. *;
   %***************************************************************************;
   %let _lsafMASTERJobDATA_tempDS_=%str(__LSAF_SASMacro_processjobtemp__);
   %let _lsaf_tempDS_=%str(__LSAF_temp__);

   %if (%sysfunc(exist(&_lsafMASTERJobDATA_tempDS_)) > 0) %then
   %do;
      %put NOTE: &macroName: Deleting temporary data set &_lsafMASTERJobDATA_tempDS_ before processing.;
      proc datasets library=work nolist;
         delete &_lsafMASTERJobDATA_tempDS_;
      quit;
      run;
   %end;

   %if (%sysfunc(exist(&_lsaf_tempDS_)) > 0) %then
   %do;
      %put NOTE: &macroName: Deleting temporary data set &_lsaf_tempDS_ before processing.;
      proc datasets library=work nolist;
         delete &_lsaf_tempDS_;
      quit;
      run;
   %end;

   %local _lsaf_tempDS_
          datasetstructure
          datasetvars
          notprovidedstring
          updating
   ;

   %let datasetvars = %str(source name value version pLabel type includeSubFolders includeFiles);
   %let datasetstructure =
      attrib source              length=$20
             name                length=$256
             value               length=$2048
             version             length=$32
             pLabel              length=$256
             type                length=$32
             includeSubFolders   length=8
             includeFiles        length=8

      ;
   data &_lsafMASTERJobDATA_tempDS_;
      &datasetstructure;
      STOP;
   run;

   %*********************************************************************************;
   %* JOB INFO data set - if supplied, validate it and add it to master data set.   *;
   %*********************************************************************************;
   %if ( &sas_dsname_info NE &notprovidedstring ) %then
   %do;
      options notes;
      %put NOTE: &callingMacro.: Validating JOB INFORMATION data set: &sas_dsname_info;
      %lsaf_validate_sasdataset(callingMacro=&macroName, sas_dsn=&sas_dsname_info, required_variables=%str(name value),
                                allow_empty_datasets=&updating);
      %if (&_lsafRC_ ne 0) %then
      %do;
         %let _lsafMsg_= sas_dsname_info: &_lsafMsg_;
         %RETURN;
      %end;

      %******************************************;
      %* JOB INFO - add to master data set.     *;
      %******************************************;
      options nonotes;
      data &_lsaf_tempDS_ (keep=&datasetvars);
         &datasetstructure;
         set &sas_dsname_info;
         source="&source_info";
      run;
      proc append base=&_lsafMASTERJobDATA_tempDS_ data=&_lsaf_tempDS_;
      run;
   %end;

   %*********************************************************************************;
   %* PARAMETERS data set - if supplied, validate it and add it to master data set. *;
   %*********************************************************************************;
   %if ( &sas_dsname_parameters NE &notprovidedstring ) %then
   %do;
      options notes;
      %put NOTE: &callingMacro.: Validating JOB PARAMETERS data set: &sas_dsname_parameters;
      %lsaf_validate_sasdataset(callingMacro=&macroName, sas_dsn=&sas_dsname_parameters,
                                required_variables=%str(name label type defaultValue fileVersion includeSubFolders),
                                required_numeric_variables=%str(includeSubFolders),
                                allow_empty_datasets=&updating);
      %if (&_lsafRC_ ne 0) %then
      %do;
         %let _lsafMsg_= sas_dsname_parameters: &_lsafMsg_;
         %RETURN;
      %end;

      %*****************************************************************;
      %* PARAMETERS: Validate that no duplicate values are present.    *;
      %* Order of original data set needs to be preserved.             *;
      %*****************************************************************;
      %put NOTE: &callingMacro.: Testing for duplicate parameters;
      %let _tempPARAMETERS_tempDS_=%str(_tempPARAMETERS_tempDS_);
      %if (%sysfunc(exist(&_tempPARAMETERS_tempDS_)) > 0) %then
      %do;
         %put NOTE: %sysfunc(lowcase(&SYSMACRONAME)): Deleting temporary data set &_tempPARAMETERS_tempDS_ before processing.;
         proc datasets library=work nolist;
            delete &_tempPARAMETERS_tempDS_;
         quit;
         run;
      %end;
      proc sort data=&sas_dsname_parameters out=&_tempPARAMETERS_tempDS_;
         by name;
      run;
      data _null_;
         set &_tempPARAMETERS_tempDS_;
         by name;
         if (first.name AND NOT(last.name)) then
         do;
            call symput("_lsafRC_", "-999");
            call symput("_lsafMsg_", "sas_dsname_parameters: Data set has duplicate values for parameter name.");
            STOP;
         end;
      run;
      %if (&_lsafRC_ ne 0) %then
      %do;
         %RETURN;
      %end;

      %******************************************;
      %* PARAMETERS - add to master data set.   *;
      %******************************************;
      options nonotes;
      data &_lsaf_tempDS_ (keep=&datasetvars);
         &datasetstructure;
         set &sas_dsname_parameters;
         source="&source_parms";
         pLabel=label;
         value=defaultValue;
         version=fileVersion;
      run;

      proc append base=&_lsafMASTERJobDATA_tempDS_ data=&_lsaf_tempDS_;
      run;

   %end;

   %********************************************************************************;
   %* PROGRAMS data set - if supplied, validate it and add it to master data set.  *;
   %********************************************************************************;
   %if ( &sas_dsname_programs NE &notprovidedstring ) %then
   %do;
      options notes;
      %put NOTE: &callingMacro.: Validating JOB PROGRAMS data set: &sas_dsname_programs;
      %lsaf_validate_sasdataset(callingMacro=&macroName, sas_dsn=&sas_dsname_programs, required_variables=%str(programPath programVersion),
                                allow_empty_datasets=&updating);
      %if (&_lsafRC_ ne 0) %then
      %do;
         %let _lsafMsg_= sas_dsname_programs: &_lsafMsg_;
         %RETURN;
      %end;

      %**************************************;
      %* PROGRAMS - add to master data set. *;
      %**************************************;
      options nonotes;
      data &_lsaf_tempDS_ (keep=&datasetvars);
         &datasetstructure;
         set &sas_dsname_programs (rename=(programPath=value programVersion=version));
         source="&source_programs";
      run;
      proc append base=&_lsafMASTERJobDATA_tempDS_ data=&_lsaf_tempDS_;
      run;
   %end;

   %********************************************************************************;
   %* SETUP data set - if supplied, validate it and add it to master data set.     *;
   %********************************************************************************;
   %if ( &sas_dsname_setup NE &notprovidedstring ) %then
   %do;
      options notes;
      %put NOTE: &callingMacro.: Validating JOB SETUP data set: &sas_dsname_setup;
      %lsaf_validate_sasdataset(callingMacro=&macroName, sas_dsn=&sas_dsname_setup, required_variables=%str(setupPath setupType 
                                setupVersion setupIncludeSubFolders setupIncludeFiles),
                                required_numeric_variables=%str(setupIncludeSubFolders setupIncludeFiles),
                                allow_empty_datasets=&updating);
      %if (&_lsafRC_ ne 0) %then
      %do;
         %let _lsafMsg_= sas_dsname_setup: &_lsafMsg_;
         %RETURN;
      %end;

      %****************************************************************;
      %* SETUP: Validate that no duplicate values are present.        *;
      %* Order of original data set needs to be preserved.            *;
      %****************************************************************;
      %put NOTE: &callingMacro.: Testing for duplicate setup paths;
      %let _tempSETUP_tempDS_=%str(_tempSETUP_tempDS_);
      %if (%sysfunc(exist(&_tempSETUP_tempDS_)) > 0) %then
      %do;
         %put NOTE: %sysfunc(lowcase(&SYSMACRONAME)): Deleting temporary data set &_tempSETUP_tempDS_ before processing.;
         proc datasets library=work nolist;
            delete &_tempSETUP_tempDS_;
         quit;
         run;
      %end;
      proc sort data=&sas_dsname_setup out=&_tempSETUP_tempDS_;
         by setupPath;
      run;
      data _null_;
         set &_tempSETUP_tempDS_;
         by setupPath;
         if (first.setupPath AND NOT(last.setupPath)) then
         do;
            call symput("_lsafRC_", "-999");
            call symput("_lsafMsg_", "sas_dsname_setup: Data set has duplicate values for setupPath.");
            STOP;
         end;
      run;
      %if (&_lsafRC_ ne 0) %then
      %do;
         %RETURN;
      %end;

      %*****************************************;
      %* SETUP - add to master data set.       *;
      %*****************************************;
      options nonotes;
      data &_lsaf_tempDS_ (keep=&datasetvars);
         &datasetstructure;
         set &sas_dsname_setup (rename=(setupPath=value setupVersion=version setupType=type setupIncludeSubFolders=includeSubFolders
           setupIncludeFiles=includeFiles));
         source="&source_setup";
      run;
      proc append base=&_lsafMASTERJobDATA_tempDS_ data=&_lsaf_tempDS_;
      run;
   %end;

   %****************************************************************************************;
   %* CHECKIN LOCATIONS data set - if supplied, validate it and add it to master data set. *;
   %****************************************************************************************;
   %if ( &sas_dsname_checkin NE &notprovidedstring ) %then
   %do;
      options notes;
      %put NOTE: &callingMacro.: Validating JOB CHECKIN LOCATIONS data set: &sas_dsname_checkin;
      %lsaf_validate_sasdataset(callingMacro=&macroName, sas_dsn=&sas_dsname_checkin, required_variables=%str(checkinPath checkinIncludeSubFolders),
                                required_numeric_variables=%str(checkinIncludeSubFolders),
                                allow_empty_datasets=&updating);
      %if (&_lsafRC_ ne 0) %then
      %do;
         %let _lsafMsg_= sas_dsname_checkin: &_lsafMsg_;
         %RETURN;
      %end;

      %*****************************************************************;
      %* CHECKIN: Validate that no duplicate values are present.       *;
      %* Order of original data set needs to be preserved.             *;
      %*****************************************************************;
      %put NOTE: &callingMacro.: Testing for duplicate checkin location paths;
      %let _tempCHECKIN_tempDS_=%str(_tempCHECKIN_tempDS_);
      %if (%sysfunc(exist(&_tempCHECKIN_tempDS_)) > 0) %then
      %do;
         %put NOTE: %sysfunc(lowcase(&SYSMACRONAME)): Deleting temporary data set &_tempCHECKIN_tempDS_ before processing.;
         proc datasets library=work nolist;
            delete &_tempCHECKIN_tempDS_;
         quit;
         run;
      %end;
      proc sort data=&sas_dsname_checkin out=&_tempCHECKIN_tempDS_;
         by checkinPath;
      run;
      data _null_;
         set &_tempCHECKIN_tempDS_;
         by checkinPath;
         if (first.checkinPath AND NOT(last.checkinPath)) then
         do;
            call symput("_lsafRC_", "-999");
            call symput("_lsafMsg_", "sas_dsname_checkin: Data set has duplicate values for checkinPath.");
            STOP;
         end;
      run;
      %if (&_lsafRC_ ne 0) %then
      %do;
         %RETURN;
      %end;

      %******************************************;
      %* CHECKIN - add to master data set.       *;
      %******************************************;
      options nonotes;
      data &_lsaf_tempDS_ (keep=&datasetvars);
         &datasetstructure;
         set &sas_dsname_checkin (rename=(checkinPath=value checkinIncludeSubFolders=includeSubFolders));
         source="&source_checkin";
      run;
      proc append base=&_lsafMASTERJobDATA_tempDS_ data=&_lsaf_tempDS_;
      run;
   %end;

   %let _lsafRC_=0;
   
   %***************************************************************************;
   %* Cleanup temp data sets.                                                 *;
   %***************************************************************************;

   %if (%sysfunc(exist(&_lsaf_tempDS_)) > 0) %then
   %do;
      proc datasets library=work nolist;
         delete &_lsaf_tempDS_;
      quit;
      run;
   %end;
   
   %if (%sysfunc(exist(&_tempSETUP_tempDS_)) > 0) %then
   %do;
      proc datasets library=work nolist;
         delete &_tempSETUP_tempDS_;
      quit;
      run;
   %end;
   
   %if (%sysfunc(exist(&_tempCHECKIN_tempDS_)) > 0) %then
   %do;
      proc datasets library=work nolist;
         delete &_tempCHECKIN_tempDS_;
      quit;
      run;
   %end;
   
   %if (%sysfunc(exist(&_tempPARAMETERS_tempDS_)) > 0) %then
   %do;
      proc datasets library=work nolist;
         delete &_tempPARAMETERS_tempDS_;
      quit;
      run;
   %end;

   options &notesOption;

%mend lsaf_process_jobdatasets;
