%* lsaf_logout                                                                                   *;
%*                                                                                               *;
%* Closes an open connection to SAS Life Science Analytics Framework.                            *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 1.1                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2013-06-26  version 1.2 update - trusted session                                              *;
%* 2013-08-30  updates for standardization and coding conventions                                *;
%* 2016-08-26  rebrand as LSAF                                                                   *;
%* 2020-02-07  move to AuthenticationService                                                     *;
%* 2020-07-14  update for refactor of java service hierarchy.                                    *;
%* 2020-07-29  cleanup work.mvarstable data set.                                                 *;
%* 2021-08-05  Remove header documentation                                                       *;
%*                                                                                               *;

%macro lsaf_logout(
  ) /des='Terminate the connection to SAS Life Science Analytics Framework.';

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data _null_;
    attrib returnCode      length=8;
    attrib message         length=$200;
    attrib logMsgPrefix    length=$70;
    attrib logMsgPrimary   length=$70;
    attrib logMsg          length=$300;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/authentication/AuthenticationService");
    declare javaobj results("com/sas/lsaf/macro/common/result/Result");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("logout", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode eq 0) then
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" The macro variables will be deleted: _LSAFRC_ _LSAFMSG_" ;
    end;
    else
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Unable to close connection." ;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);

  run;

  %***********************************;
  %* Macro variable cleanup          *;
  %***********************************;

  %if (&_lsafRC_ ne 0) %then %return;

  *retain original value for options and then turn source off;
  %let sourceOption=%sysfunc(getoption(source));
  %let notesOption=%sysfunc(getoption(notes));
  options nosource nonotes;

  %global lsaf_mvars;
  data _null_;
     set sashelp.vmacro end=end;
     length lsaf_mvars $1000;
     retain lsaf_mvars;
     pos=index(name,'_LSAF');
     if (pos=1) then 
     do;
       lsaf_mvars=catx(' ', lsaf_mvars, name); 
       message="Deleting macro variable " || strip(name) || " from scope " || strip(scope) || ".";
       put message;
     end; 
     if end then call symputx('lsaf_mvars', lsaf_mvars);
  run;
  %symdel &lsaf_mvars;
  %symdel lsaf_mvars;

  * reset options;
  options &sourceOption &notesOption;

%mend lsaf_logout;
