%* lsaf_isowner                                                                                  *;
%*                                                                                               *;
%* Indicates whether the specified user is the owner of an item in the SAS Life Science          *;
%* Analytics Framework repository.                                                               *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 2.5                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2019-02-05  initial coding                                                                    *;
%* 2020-07-13  update for refactor of java service hierarchy.                                    *;
%* 2021-08-05  Remove header documentation                                                       *;
%*                                                                                               *;

%macro lsaf_isowner(
  lsaf_path=,
  lsaf_acltype=,
  lsaf_userid=
  ) /des='Is the specified user the owner of the item?';

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafIsOwner_
          _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data _null_;
    attrib returnCode          length=8;
    attrib message             length=$200;
    attrib logMsgPrefix        length=$70;
    attrib logMsg              length=$300;
    attrib isOwner             length=3;
    attrib logMsgPrimary       length=$70;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/acl/AclService");
    declare javaobj results("com/sas/lsaf/macro/common/result/Result");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticBooleanMethod("isOwner", "%bquote(&lsaf_path)","%bquote(&lsaf_acltype)","%bquote(&lsaf_userid)", results, isOwner);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot determine if the user &lsaf_userid is the &lsaf_acltype owner of item: &lsaf_path." ;
      isOwner=-1;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      if (isOwner) 
        then logMsg=" The user &lsaf_userid is the &lsaf_acltype owner of item: &lsaf_path.";
        else logMsg=" The user &lsaf_userid is not the &lsaf_acltype owner of item: &lsaf_path.";
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    put logMsgPrefix " _lsafMsg_= " message;
    put logMsgPrefix " _lsafRC_= " returnCode;
    put logMsgPrefix " _lsafIsOwner_= " isOwner;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafIsOwner_", isOwner);

  run;

%mend lsaf_isowner;
