%* lsaf_importstudyvlm                                                                           *;
%*                                                                                               *;
%* Imports study value level metadata from the specified data set and optionally creates a       *;
%* validation report as a .csv file in the designated location.                                  *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 2.4                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2020-09-25  initial coding                                                                    *;
%* 2021-08-16  Remove header documentation                                                       *;
%*                                                                                               *;

%macro lsaf_importstudyvlm(
  lsaf_studyid=,
  lsaf_basestandardname=,
  lsaf_modelid=,
  lsaf_importaction=, 
  lsaf_location=, 
  lsaf_vlmpath=,
  lsaf_vlmversion=,
  lsaf_reportlocation=,
  lsaf_reportpath=,
  lsaf_reportoverwrite=,
  lsaf_reportenableversioning=, 
  lsaf_reportversiontype=,
  lsaf_reportcustomversion=,
  lsaf_reportcomment=,
  sas_result_dsname=work.lsafimportstudyvlm
  ) /des='Imports study value level metadata to a data set in the repository.';
  
  %if "&sas_result_dsname" EQ "" %then %let sas_result_dsname=work.lsafimportstudyvlm;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_result_dsname (label="Created by &SYSMACRONAME" keep=studyId name value);
    attrib returnCode           length=8
           message              length=$200
           logMsgPrefix         length=$70
           logMsgPrimary        length=$70
           logMsg               length=$300
           rowCount             length=8
           studyId              length=$64      label="Study Id"
           name                 length=$64      label="Property Name"
           value                length=$256     label="Property Value"
           completionStatus     length=$32
           completionStatusMsg  length=$256
           sasLogFlag           length=$200
           sasLogMsg            length=$200 
    ;

    %*****************************;
    %* Declare the java objects  *;
    %*****************************;
    declare javaobj srv("com/sas/lsaf/macro/clinical/study/StudyService") ;
    declare javaobj importParameters("com/sas/lsaf/macro/clinical/SasClinicalUpdateParameters");
    declare javaobj reportFileParameters("com/sas/lsaf/macro/content/SasFileCreateParameters"); 
    declare javaobj results("com/sas/lsaf/macro/clinical/study/SasStudyImportValidationResult");
 
    %***********************************;
    %* Attempt the operation           *;
    %***********************************;

    importParameters.callVoidMethod("updateImportData", "%bquote(&lsaf_location)", "%bquote(&lsaf_vlmpath)", "%bquote(&lsaf_vlmversion)", 
          "%bquote(&lsaf_importaction)");
		
    reportFileParameters.callVoidMethod("updateData", "%bquote(&lsaf_reportlocation)", "%bquote(&lsaf_reportpath)", "%bquote(&lsaf_reportoverwrite)", 
          "%bquote(&lsaf_reportenableversioning)", "%bquote(&lsaf_reportversiontype)", "%bquote(&lsaf_reportcustomversion)", "%bquote(&lsaf_reportcomment)");

    srv.callStaticVoidMethod("importStudyValueLevelMetadata", "%bquote(&lsaf_studyid)", "%bquote(&lsaf_basestandardname)", "%bquote(&lsaf_modelid)", 
          importParameters, reportFileParameters, results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);
    results.callStringMethod("getCompletionStatus", completionStatus);
    results.callStringMethod("getCompletionStatusMessage", completionStatusMsg);
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      if (returnCode in (-400:-300)) then 
      do;
        logMsgPrimary= strip(completionStatus) || ": &_LSAF_PRODUCT_NAME_ Macro:";
        logMsg=" " || strip(completionStatusMsg) ;
      end;
      else 
      do;
        logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
        logMsg=" Could not import the study value level metadata." ;
      end;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Returned import completion information for the study value level metadata.";
    end;
    studyId="&lsaf_studyid";
    results.callIntMethod("getSize", rowCount);
    do row=0 to rowCount - 1;
      results.callStringMethod("getPropertyName", row, name);
      results.callStringMethod("getPropertyValue", row, value);
      output;
    end;
    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount<=0) 
      then put logMsgPrefix " The study value level metadata were not imported.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_result_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    results.callIntMethod("getValidationMessagesSize", rowCount);
    if (rowCount>0) then 
    do;
      macroNameStamp=strip(lowcase("&SYSMACRONAME")) || ": ";
      do row=0 to rowCount - 1;
        results.callStringMethod("getMessageSeverity", row, sasLogFlag);
        results.callStringMethod("getMessageDetailedProblem", row, sasLogMsg);
        sasLogFlag = strip(sasLogFlag) || ": ";
        sasLogMsg  = strip(sasLogMsg);
        put sasLogFlag macroNameStamp sasLogMsg;
      end;
      put "-" ;
    end;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);
  run;

  %***********************************;
  %* DO NOT Sort the data set        *;
  %***********************************;

%mend lsaf_importstudyvlm;
