%* lsaf_importstudytlfs                                                                          *;
%*                                                                                               *;
%* Imports study tables, figures, and listings from the specified data sets and optionally       *;
%* creates a validation report as a .csv file in the specified location.                         *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 2.4                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2020-09-22  initial coding                                                                    *;
%* 2021-08-16  Remove header documentation                                                       *;
%*                                                                                               *;

%macro lsaf_importstudytlfs(
  lsaf_studyId=, 
  lsaf_basestandardname=,
  lsaf_modelId=,
  lsaf_importAction=, 
  lsaf_location=, 
  lsaf_sectionsPath=,
  lsaf_sectionsVersion=,
  lsaf_tlfsPath=,
  lsaf_tlfsVersion=,
  lsaf_resultsPath=,
  lsaf_resultsVersion=,
  lsaf_datasetsPath=,
  lsaf_datasetsVersion=,
  lsaf_reportLocation=,
  lsaf_reportPath=,
  lsaf_reportOverwrite=,
  lsaf_reportEnableVersioning=, 
  lsaf_reportVersionType=,
  lsaf_reportCustomVersion=,
  lsaf_reportComment=,
  sas_result_dsname=work.lsafimportstudytlfs
  ) /des='Imports study tables, figures, and listings to data sets in the repository.';
  
  %if "&sas_result_dsname" EQ "" %then %let sas_result_dsname=work.lsafimportstudytlfs;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_result_dsname (label="Created by &SYSMACRONAME" keep=studyId name value);
    attrib returnCode           length=8
           message              length=$200
           logMsgPrefix         length=$70
           logMsgPrimary        length=$70
           logMsg               length=$300
           rowCount             length=8
           studyId              length=$64      label="Study Id"
           name                 length=$64      label="Property Name"
           value                length=$256     label="Property Value"
           completionStatus     length=$32
           completionStatusMsg  length=$256
           sasLogFlag           length=$200
           sasLogMsg            length=$200 
    ;

    %*****************************;
    %* Declare the java objects  *;
    %*****************************;
    declare javaobj srv("com/sas/lsaf/macro/clinical/study/StudyService") ;
    declare javaobj importParameters("com/sas/lsaf/macro/clinical/SasClinicalUpdateTlfsParameters",
		"%bquote(&lsaf_location)", "%bquote(&lsaf_importaction)", 
		"%bquote(&lsaf_sectionsPath)", "%bquote(&lsaf_sectionsVersion)", "%bquote(&lsaf_tlfsPath)", "%bquote(&lsaf_tlfsVersion)", 
		"%bquote(&lsaf_resultsPath)", "%bquote(&lsaf_resultsVersion)", "%bquote(&lsaf_datasetsPath)", "%bquote(&lsaf_datasetsVersion)"); 
    declare javaobj exportData("com/sas/lsaf/macro/content/SasFileCreateParameters"); 
    declare javaobj results("com/sas/lsaf/macro/clinical/study/SasStudyImportValidationResult");
 
    %***********************************;
    %* Attempt the operation           *;
    %***********************************;		
    exportData.callVoidMethod("updateData", "%bquote(&lsaf_reportLocation)", "%bquote(&lsaf_reportPath)", "%bquote(&lsaf_reportOverwrite)", 
                "%bquote(&lsaf_reportEnableVersioning)", "%bquote(&lsaf_reportVersionType)", "%bquote(&lsaf_reportCustomVersion)", "%bquote(&lsaf_reportComment)");

    srv.callStaticVoidMethod("importStudyTlfs", "%bquote(&lsaf_studyId)", "%bquote(&lsaf_baseStandardName)", "%bquote(&lsaf_modelId)", importParameters, exportData, results); * <---------------- CHANGED THIS (AGAIN).;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);
    results.callStringMethod("getCompletionStatus", completionStatus);
    results.callStringMethod("getCompletionStatusMessage", completionStatusMsg);
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      if (returnCode in (-400:-300)) then 
      do;
        logMsgPrimary= strip(completionStatus) || ": &_LSAF_PRODUCT_NAME_ Macro:";
        logMsg=" " || strip(completionStatusMsg) ;
      end;
      else 
      do;
        logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
        logMsg=" Could not import the study tables, listings, and figures." ;
      end;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Returned import completion information for the study tables, listings, and figures.";
    end;
    studyId="&lsaf_studyId";
    results.callIntMethod("getSize", rowCount);
    do row=0 to rowCount - 1;
      results.callStringMethod("getPropertyName", row, name);
      results.callStringMethod("getPropertyValue", row, value);
      output;
    end;
    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount<=0) 
      then put logMsgPrefix " The study tables, listings, and figures were not imported.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_result_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    results.callIntMethod("getValidationMessagesSize", rowCount);
    if (rowCount>0) then 
    do;
      macroNameStamp=strip(lowcase("&SYSMACRONAME")) || ": ";
      do row=0 to rowCount - 1;
        results.callStringMethod("getMessageSeverity", row, sasLogFlag);
        results.callStringMethod("getMessageDetailedProblem", row, sasLogMsg);
        sasLogFlag = strip(sasLogFlag) || ": ";
        sasLogMsg  = strip(sasLogMsg);
        put sasLogFlag macroNameStamp sasLogMsg;
      end;
      put "-" ;
    end;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);
  run;

  %***********************************;
  %* DO NOT Sort the data set        *;
  %***********************************;

%mend lsaf_importstudytlfs;
