%* lsaf_getworkspacejobsetup                                                                      *;
%*                                                                                                *;
%* Gets the metadata for the setup items for the specified job in the workspace of the current    *;
%* user. The metadata is stored in a SAS data set. The list does not include the setup programs.  *;
%*                                                                                                *;
%* macrotype LSAF                                                                                 *;
%* since 1.5                                                                                      *;
%* exposure external                                                                              *;
%*                                                                                                *;
%* History                                                                                        *;
%* 2016-10-04  initial coding.                                                                    *;
%* 2019-02-11  update to 2.1                                                                      *;
%* 2019-11-01  update to handle a blank SAS output file.                                          *;
%* 2020-06-25  update to handle includeFiles column in SAS input data set.                        *;
%* 2020-08-11  renamed macro from lsaf_getworkspacejobinputs for LSAF 5.3 compatibility.          *;
%* 2021-04-07  update for java code refactor                                                      *;
%* 2021-08-06  Remove header documentation                                                        *;
%*                                                                                                *;

%macro lsaf_getworkspacejobsetup(
   lsaf_path=,
   sas_dsname=work.lsafGetWorkspaceJobSetup
   ) /des='Get the list of setup items for SAS Life Science Analytics Framework workspace job.';
  
   %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetWorkspaceJobSetup;

   %****************************************;
   %* Initialize macro variables           *;
   %****************************************;
  %lsaf_getProductName();

   %global _lsafMsg_
           _lsafRC_
   ;

   %let _lsafRC_ =%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

   %**********************************************;
   %* Start the datastep to call the java code   *;
   %**********************************************;
   data &sas_dsname (label="Created by &SYSMACRONAME"
                     keep=jobPath jobVersion setupType setupPath setupVersion setupIncludeSubFolders setupIncludeFiles
   );

   attrib returnCode      length=8
          message         length=$200
          logMsgPrefix    length=$70
          logMsgPrimary   length=$70
          logMsg          length=$300
          rowCount        length=8

          jobPath                 length=$2048  label="Job Path"
          jobVersion              length=$32    label="Job Version"
          setupPath               length=$2048  label="Setup Path"
          setupType               length=$32    label="Setup Type"
          setupVersion            length=$32    label="Setup File Version"
          setupIncludeSubFolders  length=8      label="Include Setup Container SubFolders"
          includeSubFolders       length=$5     label="FOR INTERNAL PROCESSING ONLY.  WILL BE DROPPED."
          setupIncludeFiles       length=8      label="Include Setup Files"
          includeFiles            length=$5     label="FOR INTERNAL PROCESSING ONLY.  WILL BE DROPPED."
   ;

   rowCount=0;

   %***********************************;
   %* Declare the java objects        *;
   %***********************************;
   declare javaobj srv("com/sas/lsaf/macro/execution/job/JobService") ;
   declare javaobj results("com/sas/lsaf/macro/execution/job/SasJobPublishSetupsResult");

   %***********************************;
   %* Attempt the operation           *;
   %***********************************;
   srv.callStaticVoidMethod("getWorkspaceJobSetup", "%bquote(&lsaf_path)", results) ;

   %***********************************;
   %* Retrieve the results            *;
   %***********************************;
   results.callIntMethod("getReturnCode", returnCode);
   results.callStringMethod("getReturnMessage", message);

   %***********************************;
   %* Process the results             *;
   %***********************************;
   logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
   if (returnCode ne 0) then
   do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve setup information for job &lsaf_path." ;
   end;
   else
   do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Setup information was returned for job &lsaf_path" ;
      results.callStringMethod("getJobPath", jobPath);
      results.callStringMethod("getJobVersion", jobVersion);
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
         results.callStringMethod("getPath", row, setupPath);
         results.callStringMethod("getType", row, setupType);
         results.callStringMethod("getFileVersion", row, setupVersion);
         results.callStringMethod("getIncludeSubFolders", row, includeSubFolders);
         setupIncludeSubFolders=input(includeSubFolders,best.);
         results.callStringMethod("getIncludeFiles", row, includeFiles);
         setupIncludeFiles=input(includeFiles,best.);
         output;
      end;
   end;

   %***********************************;
   %* Print messages in log           *;
   %***********************************;
   put;
   put logMsgPrimary " " logMsg;
   put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
   if (rowCount=0) 
     then put logMsgPrefix " No setup information was found.";
     else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
   put logMsgPrefix " _lsafMsg_= " message ;
   put logMsgPrefix " _lsafRC_= " returnCode ;
   put;

   %***********************************;
   %* Set the macro variables         *;
   %***********************************;
   call symputx("_lsafRC_", returnCode);
   call symputx("_lsafMsg_", message);

   run;

%mend lsaf_getworkspacejobsetup;
