%* lsaf_getworkspacejobparameters                                                                 *;
%*                                                                                                *;
%* Gets the job parameter metadata for the specified job in the workspace of the current user.    *;
%* The metadata is stored in a SAS data set.                                                      *;
%*                                                                                                *;
%* macrotype LSAF                                                                                 *;
%* since 1.5                                                                                      *;
%* exposure external                                                                              *;
%*                                                                                                *;
%* History                                                                                        *;
%* 2016-10-04  initial coding.                                                                    *;
%* 2019-02-11  update to 2.1.                                                                     *;
%* 2019-11-01  update to handle a blank SAS output file.                                          *;
%* 2021-08-06  Remove header documentation                                                        *;
%*                                                                                                *;

%macro lsaf_getworkspacejobparameters(
   lsaf_path=,
   sas_dsname=work.lsafGetWorkspaceJobParameters
   ) /des='Get the list of the parameters for SAS Life Science Analytics Framework workspace job.';
  
   %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetWorkspaceJobParameters;

   %****************************************;
   %* Initialize macro variables           *;
   %****************************************;
   %lsaf_getProductName();

   %global _lsafMsg_
           _lsafRC_
   ;

   %let _lsafRC_ =%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);


   %**********************************************;
   %* Start the datastep to call the java code   *;
   %**********************************************;
   data &sas_dsname (label="Created by &SYSMACRONAME"
                     keep=jobPath jobVersion name label type defaultValue fileVersion includeSubFolders
   );
   attrib returnCode         length=8
          message            length=$200
          logMsgPrefix       length=$70
          logMsgPrimary      length=$70
          logMsg             length=$300
          rowCount           length=8

          jobPath            length=$2048      label="Job Path"
          jobVersion         length=$32        label="Job Version"
          name               length=$256       label="Parameter Name"
          label              length=$256       label="Parameter Label"
          type               length=$32        label="Parameter Type"
          defaultValue       length=$2048      label="Parameter Default Value"
          fileVersion        length=$32        label="File Version if File Type"
          includeSubFolders  length=8          label="Include Container SubFolders"
          includesSubFolders length=$5         label="FOR INTERNAL PROCESSING ONLY.  WILL BE DROPPED."
   ;

   rowCount=0;

   %***********************************;
   %* Declare the java objects        *;
   %***********************************;
   declare javaobj srv("com/sas/lsaf/macro/execution/job/JobService") ;
   declare javaobj results("com/sas/lsaf/macro/execution/job/SasJobParametersResult");

   %***********************************;
   %* Attempt the operation           *;
   %***********************************;
   srv.callStaticVoidMethod("getWorkspaceJobParameters", "%bquote(&lsaf_path)", results) ;

   %***********************************;
   %* Retrieve the results            *;
   %***********************************;
   results.callIntMethod("getReturnCode", returnCode);
   results.callStringMethod("getReturnMessage", message);

   %***********************************;
   %* Process the results             *;
   %***********************************;
   logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
   if (returnCode ne 0) then
   do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve parameters for job &lsaf_path." ;
   end;
   else
   do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Parameters were returned for job &lsaf_path" ;
      results.callStringMethod("getJobPath", jobPath);
      results.callStringMethod("getJobVersion", jobVersion);
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
         results.callStringMethod("getName", row, name);
         results.callStringMethod("getLabel", row, label);
         results.callStringMethod("getType", row, type);
         results.callStringMethod("getDefaultValue", row, defaultValue);
         results.callStringMethod("getFileVersion", row, fileVersion);
         results.callStringMethod("getContainerIncludeSubfolders", row, includesSubFolders);
         includeSubFolders=input(includesSubFolders,best.);
         output;
      end;
   end;

   %***********************************;
   %* Print messages in log           *;
   %***********************************;
   put;
   put logMsgPrimary " " logMsg;
   put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
   if (rowCount=0) 
     then put logMsgPrefix " No parameters were found.";
     else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
   put logMsgPrefix " _lsafMsg_= " message ;
   put logMsgPrefix " _lsafRC_= " returnCode ;
   put;

   %***********************************;
   %* Set the macro variables         *;
   %***********************************;
   call symputx("_lsafRC_", returnCode);
   call symputx("_lsafMsg_", message);

   run;

%mend lsaf_getworkspacejobparameters;
