%* lsaf_getworkspacechildren                                                                     *;
%*                                                                                               *;
%* Gets the metadata for items in a folder in the workspace of the current user. The metadata is *;
%* stored in a SAS data set.                                                                     *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 1.5                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2016-11-09  initial coding                                                                    *;
%* 2018-01-09  updates LSAF 5.1 functionality                                                    *;
%* 2019-11-01  update to handle a blank SAS output file.                                         *;
%* 2020-07-13  update for refactor of java service hierarchy.                                    *;
%* 2020-10-22  update to include item size in output data set.                                   *;
%* 2020-11-11  add formattedSize and update to set a -1 value for file size to missing.          *;
%* 2021-08-18  Remove header documentation                                                       *;
%* 2021-08-31  DE16792                                                                           *;
%* 2021-09-20  Remove parameter defaults (DE16775)                                               *;
%* 2021-10-12  US11739                                                                           *;
%* 2024-07-26  2.7: Notes option updates.                                                        *;
%*                                                                                               *;

%macro lsaf_getworkspacechildren(
  lsaf_path=,
  sas_dsname=work.lsafGetWorkspaceChildren,
  lsaf_recursive=
  ) /des='Get a list of the items in a folder in SAS Life Science Analytics Framework workspace.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetWorkspaceChildren;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_
         notesOption
  ;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=path name itemType isFolder size formattedsize lastModified dateLastModified);

    attrib returnCode          length=8
           message             length=$200
           logMsgPrefix        length=$70
           logMsgPrimary       length=$70
           logMsg              length=$300
           rowCount            length=8

           path                length=$2048    label="Item Path"
           name                length=$256     label="Item Name"
           itemType            length=$32      label="Item Type"
           isFolder            length=8        label="Is item A Container?"
           size                length=8        label="Size In Bytes"
           formattedsize       length=$20      label="Size (Formatted)"
           lastModified        length=$64      label="Last Content Modification Date"
           dateLastModified                    label="Date the item's content was last modified (SAS DateTime Format)"
           lastModifiedTempSAS length=$64      label="TEMPORARY - WILL BE DROPPED"
    ;

    format dateLastModified datetime.;
    rowCount=0;

    %***********************************;
    %* Declare the java items        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/content/workspace/WorkspaceService");
    declare javaobj results("com/sas/lsaf/macro/content/workspace/SasWorkspaceItemsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getChildren", "%bquote(&lsaf_path)", "%bquote(&lsaf_recursive)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not retrieve list of items from %bquote(&lsaf_path)" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Returned a list of workspace items in %bquote(&lsaf_path)";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getPath", row, path);
        results.callStringMethod("getName", row, name);
        results.callIntMethod("isFolder", row, isFolder);
        results.callLongMethod("getFileSize", row, size);
        if size < 0 then size=.;
        results.callStringMethod("getFormattedFileSize", row, formattedsize); 
        results.callStringMethod("getItemType", row, itemType);
        results.callStringMethod("getLastModified", row, lastModified);
        results.callStringMethod("getLastModifiedSasFormat", row, lastModifiedTempSas);

        dateLastModified=input(lastModifiedTempSas, datetime.);

        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No children were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put logMsgPrefix " lsaf_recursive= &lsaf_recursive" ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by path;
    run;
    options &notesOption.;
  %end;

%mend lsaf_getworkspacechildren;
