%* lsaf_getversions                                                                               *;
%*                                                                                                *;
%* Gets the metadata for all of the versions of a file. The metadata is stored in a SAS data set. *;
%*                                                                                                *;
%* macrotype LSAF                                                                                 *;
%* since 1.2                                                                                      *;
%* exposure external                                                                              *;
%*                                                                                                *;
%* History                                                                                        *;
%* 2013-08-30  updates for standardization and coding conventions                                 *;
%* 2014-05-07  updated documentation and fixed data set processing.                               *;
%* 2016-08-26  rebrand as LSAF                                                                    *;
%* 2016-01-23  updates for LSAF 5.1                                                               *;
%* 2019-11-01  update to handle a blank SAS output file.                                          *;
%* 2021-08-13  Remove header documentation                                                        *;
%* 2021-08-31  DE16792                                                                            *;
%* 2021-10-12  US11739                                                                            *;
%*                                                                                                *;

%macro lsaf_getversions(
  lsaf_path=,
  sas_dsname=work.lsafGetVersions
  ) /des='Get a list of all the versions of an item in the SAS Life Science Analytics Framework in the repository.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetVersions;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=path version isLatest isSigned size comment createdBy created dateCreated);

    attrib returnCode                    length=8
           message                       length=$200
           logMsgPrefix                  length=$70
           logMsgPrimary                 length=$70
           logMsg                        length=$300
           rowCount                      length=8

           path                          length=$2048    label="Item Path"
           version                       length=$32      label="Version Number"
           isSigned                      length=8        label="Is Version Signed"
           isLatest                      length=8        label="Is Latest Version"
           size                          length=8        label="Size in Bytes"
           createdBy                     length=$64      label="Userid that created the version"
           created                       length=$64      label="Version Creation Date"
           dateCreated                   length=$64      label="Date the item was versioned (SAS DateTime Format)" 
           comment                       length=$1024    label="Comment"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java items        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/content/repository/RepositoryService") ;
    declare javaobj results("com/sas/lsaf/macro/content/repository/SasRepositoryVersionsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getVersions", "%bquote(&lsaf_path)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of versions for &lsaf_path" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of versions were returned for &lsaf_path" ;
      results.callIntMethod("getSize", rowCount);
      path = "&lsaf_path";
      do row=0 to rowCount - 1;
        results.callStringMethod("getVersion", row, version);
        results.callIntMethod("isLatest", row, isLatest);
        results.callIntMethod("isSigned", row, isSigned);
        results.callLongMethod("getSize", row, size);
        results.callStringMethod("getCreated", row, created);
        results.callStringMethod("getCreatedBy", row, createdBy);
        results.callStringMethod("getComment", row, comment);
        results.callStringMethod("getCreatedSasFormat", row, dateCreated);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No versions were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

   run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by path;
    run;
    options &notesOption;
  %end;

%mend lsaf_getversions;
