%* lsaf_getusertransutilization                                                                     *;
%*                                                                                                  *;
%* Gets the resource utilization information for the users transient workspace with sessions.       *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.5                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2021-04-07  initial programming                                                                  *;
%* 2021-08-13  Remove header documentation                                                          *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                                   *;
%* 2024-08-20  US11873 - Add unformatted size and added precision to format.                        *;
%*                                                                                                  *;

%macro lsaf_getusertransutilization(
  sas_dsname=work.lsafgetusertransutilization
  ) /des='Gets the resource utilization information for the users transient workspace with sessions.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetusertransutilization;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;
  %local notesOption;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=owner sessionState clientId sessionLabel size unformattedsize);

    attrib returnCode       length=8
           message          length=$200
           logMsgPrefix     length=$70
           logMsgPrimary    length=$70
           logMsg           length=$300
           rowCount         length=8

           owner            length=$64   label="Owner User ID"
           sessionState     length=$32   label="Session State"
           clientId         length=8     label="Incremental Number Assigned to Session"
           sessionLabel     length=$64   label="Session Label"
           size             length=8     label="Session Size - Formatted"    format=sizekmg12.2
           unformattedsize  length=8     label="Session Size - Unformatted"
    ;

    rowCount=0;
    
    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    
    declare javaobj srv("com/sas/lsaf/macro/content/utilization/ResourceUtilizationService") ;
    declare javaobj results("com/sas/lsaf/macro/content/utilization/SasUserTransientUtilizationInfosResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getUserTransientWorkspaceUtilizationDetails", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve transient workspace resource utilization information." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Get transient workspace resource utilization information ran successfully." ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getOwner", row, owner);
        results.callStringMethod("getSessionState", row, sessionState);
        results.callIntMethod("getSessionClientId", row, clientId);
        results.callStringMethod("getSessionSubmissionName", row, sessionLabel);
        results.callLongMethod("getUtilizationSize", row, size);
		unformattedsize=size;
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No transient workspace resource utilizations were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);
  run;
      
  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by descending size;
    run;
    options &notesOption.;
  %end;

%mend lsaf_getusertransutilization;
