%* lsaf_getuserswithglobalprivilege                                                                 *;
%*                                                                                                  *;
%* Gets all of the user accounts that have the specified privilege identifier. The properties are   *;
%* stored in the SAS data set LSAFGETUSERSWITHGLOBALPRIVILEGE.                                      *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.3                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2020-03-18  initial coding                                                                       *;
%* 2020-07-13  update for refactor of java service hierarchy.                                       *;
%* 2021-08-05  Remove header documentation                                                          *;
%* 2021-10-13  US11739                                                                              *;
%* 2021-12-06  US11669 Updated variable names                                                       *;
%*                                                                                                  *;

%macro lsaf_getuserswithglobalprivilege(
  lsaf_privilegeid=,
  sas_dsname=work.lsafgetuserswithglobalprivilege
  ) /des='Get a list of users with global privilege.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetuserswithglobalprivilege;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=  privilegeType privilegeId privilegeName userId displayName
  );

    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300
           rowCount        length=8

           privilegeType   length=$20    label="Privilege Type" 
           privilegeId     length=$64    label="Privilege Identifier"
           privilegeName   length=$128   label="Privilege Name"
           userId          length=$256   label="User Id"
           displayName     length=$256   label="User Display Name"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/privilege/global/GlobalPrivilegeService");
    declare javaobj results("com/sas/lsaf/macro/security/privilege/global/SasPrivilegedUsersResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getUsersWithGlobalPrivilege", "%bquote(&lsaf_privilegeid)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of users with global privileges.";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of users with global privileges was returned.";
      results.callIntMethod("getSize", rowCount);
      results.callStringMethod("getPrivilegeTypeId", privilegeType);
      results.callStringMethod("getPrivilegeId", privilegeID);
      results.callStringMethod("getPrivilegeName", privilegeName);
      do row=0 to rowCount - 1;
        results.callStringMethod("getUserId", row, userId);
        results.callStringMethod("getUserDisplayName", row, displayName);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No users with global privileges were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by userId;
    run;
    options &notesOption;
  %end;

%mend lsaf_getuserswithglobalprivilege;
