%* lsaf_getusersharedwsutilization                                                                  *;
%*                                                                                                  *;
%* Gets the user shared workspace utilization information for child folders.                        *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.7                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2024-08-26  initial programming                                                                  *;
%*                                                                                                  *;

%macro lsaf_getusersharedwsutilization(
  lsaf_path=,
  lsaf_recursive=,
  sas_dsname=work.lsafgetusersharedwsutilization
  ) /des='Gets the user shared workspace utilization information for child folders in the SAS Life Science Analytics Framework.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetuserwsutilization;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;
  %local notesOption;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=type path name size unformattedsize);

    attrib returnCode       length=8
           message          length=$200
           logMsgPrefix     length=$70
           logMsgPrimary    length=$70
           logMsg           length=$300
           rowCount         length=8

	   type             length=$32   label="Container Type"
           path             length=$2048 label="Container Path"
           name             length=$64   label="Container Name"
           size             length=8     label="Folder Size - Formatted"     format=sizekmg12.2
           unformattedsize  length=8     label="Folder Size - Unformatted"

    ;

    rowCount=0;
    
    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    
    declare javaobj srv("com/sas/lsaf/macro/content/utilization/ResourceUtilizationService") ;
    declare javaobj results("com/sas/lsaf/macro/content/utilization/SasResourceUtilizationResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getUserSharedWorkspaceUtilizationDetails", "%bquote(&lsaf_path)", "%bquote(&lsaf_recursive)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve shared workspace resource utilization information." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Get shared workspace resource utilization ran successfully." ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
	    results.callStringMethod("getTypeId", row, type);
        results.callStringMethod("getPath", row, path);
        results.callStringMethod("getName", row, name);
        results.callLongMethod("getResourceSize", row, size);
		unformattedsize=size;
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No shared workspace resource utilizations were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;
  
  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname. sortseq=linguistic(strength=primary);
      by path;
    run;
    options &notesOption.;
  %end;

%mend lsaf_getusersharedwsutilization;
