%* lsaf_getuserlicensetypes                                                                           *;
%*                                                                                                    *;
%* Gets all of the user license types that are defined. The types are stored in a SAS data set.       *;
%*                                                                                                    *;
%* macrotype LSAF                                                                                     *;
%* since 2.3                                                                                          *;
%* exposure external                                                                                  *;
%*                                                                                                    *;
%* History                                                                                            *;
%* 2019-12-11  initial coding                                                                         *;
%* 2020-12-04  updates for service refactor                                                           *;
%* 2021-08-18  Remove header documentation                                                            *;
%* 2021-10-13  US11739                                                                                *;
%*                                                                                                    *;

%macro lsaf_getuserlicensetypes(
  sas_dsname=work.lsafGetUserLicenseTypes
  ) /des='Get a list of license types.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetUserLicenseTypes;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep= licenseId name description
  );

    attrib returnCode      length=8;
    attrib message         length=$200;
    attrib logMsgPrefix    length=$70;
    attrib logMsgPrimary   length=$70;
    attrib logMsg          length=$300;
    attrib rowCount        length=8;

    attrib name            length=$256    label="License Name";
    attrib licenseId       length=$64     label="License ID";
    attrib description     length=$1024   label="License Description";

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/user/UserService") ;
    declare javaobj results("com/sas/lsaf/macro/security/user/SasUserLicenseTypesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getLicenceTypes", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of user license types.";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of user license types was returned.";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getId", row, licenseId);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getDescription", row, description);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No user license types were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by name;
    run;
    options &notesOption;
  %end;

%mend lsaf_getuserlicensetypes;