%* lsaf_getusergroups                                                                            *;
%*                                                                                               *;
%* Gets the metadata for each of the groups in which the specified user is a member. The         *;
%* metadata is stored in a SAS data set.                                                         *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 1.4                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2014-03-20  initial coding                                                                    *;
%* 2016-08-26  rebrand as LSAF                                                                   *;
%* 2019-11-01  update to handle a blank SAS output file.                                         *;
%* 2020-12-04  updates for service refactor                                                      *;
%* 2021-08-06  Remove header documentation                                                       *;
%* 2021-10-13  US11739                                                                           *;
%* 2021-10-19  principal refactor                                                                *;
%* 2021-12-03  2.6: Added path parameter                                                         *;
%*                                                                                               *;

%macro lsaf_getusergroups(
  lsaf_userid=,
  lsaf_path=,
  sas_dsname=work.lsafGetUserGroups
  ) /des='Get a list of the groups in which a user is a member.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetUserGroups;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_ notesOption;

  %let _lsafRowCount_=0;

  %**********************************************;
  %*  Start the datastep to call the java code  *;
  %**********************************************;
  data &sas_dsname (
          label="Created by &SYSMACRONAME"
          keep=userId name description grpSrcCtxt
          );
    attrib returnCode      length=8;
    attrib message         length=$200;
    attrib logMsgPrefix    length=$70;
    attrib logMsgPrimary   length=$70;
    attrib logMsg          length=$300;
    attrib rowCount        length=8;

    attrib userId          length=$256      label="User Id";
    attrib name            length=$256      label="Group Name";
    attrib description     length=$128      label="Group Description";
    attrib grpSrcCtxt      length=$2048     label="Group Context Source Path";

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/group/GroupService");
    declare javaobj results("com/sas/lsaf/macro/security/SasPrincipalsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getUserGroups", "%bquote(&lsaf_userid)", "%bquote(&lsaf_path)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve groups for &lsaf_userid." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Operation to retrieve groups was successful for &lsaf_userid." ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        userId="&lsaf_userid";
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getDescription", row, description);
        results.callStringMethod("getContextPath", row, grpSrcCtxt);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No groups were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
   %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by name grpSrcCtxt;
    run;
    options &notesOption;
  %end;

%mend lsaf_getusergroups;
