%* lsaf_getuserauthproviders                                                                        *;
%*                                                                                                  *;
%* Gets all of the user authentication providers available to the system. The providers are stored  *;
%* in a SAS data set.                                                                               *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.3                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2019-12-11  initial coding                                                                       *;
%* 2020-12-04  updates for service refactor                                                         *;
%* 2021-08-18  Remove header documentation                                                          *;
%* 2021-10-13  US11739                                                                              *;
%*                                                                                                  *;

%macro lsaf_getuserauthproviders(
  sas_dsname=work.lsafGetUserAuthProviders
  ) /des='Get a list of user authentication providers.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetUserAuthProviders;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep= providerId name description isPrimary
  );

    attrib returnCode      length=8;
    attrib message         length=$200;
    attrib logMsgPrefix    length=$70;
    attrib logMsgPrimary   length=$70;
    attrib logMsg          length=$300;
    attrib rowCount        length=8;

    attrib name            length=$256    label="Authentication Provider Name";
    attrib providerId      length=$64     label="Authentication Provider ID";
    attrib description     length=$1024   label="Authentication Provider Description";
    attrib isPrimary       length=8       label="Is Authentication Provider Primary";

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/user/UserService") ;
    declare javaobj results("com/sas/lsaf/macro/security/user/SasUserAuthenticationProvidersResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getAuthenticationProviders", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of user authentication providers.";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of user authentication providers was returned.";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getId", row, providerId);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getDescription", row, description);
        results.callBooleanMethod("isPrimary", row, isPrimary);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No user authentication providers were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by name;
    run;
    options &notesOption;
  %end;

%mend lsaf_getuserauthproviders;