%* lsaf_gettypeextendedattributes                                                               *;
%*                                                                                               *;
%* Gets the metadata for the item types for the extended attributes. The metadata is stored in   *;
%* a SAS data set.                                                                               *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 2.6                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2021-10-28  initial coding                                                                    *;
%*                                                                                               *;

%macro lsaf_gettypeextendedattributes(
  lsaf_typeid=,
  sas_dsname=work.lsafgettypeextendedattributes
  ) /des='Get a list of the extended attributes in SAS Life Science Analytics Framework.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgettypeextendedattributes;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=typeId typeName attrId attrName attrType attrIsInherited
                    );

    attrib returnCode                length=8
           message                   length=$200
           logMsgPrefix              length=$70
           logMsg                    length=$300
           rowCount                  length=8
           logMsgPrimary             length=$70

           typeId                    length=$2048    label="Type ID"
           typeName                  length=$256     label="Type Name"
           attrId                    length=$2048    label="Extended Attribute ID"
           attrName                  length=$256     label="Extended Attribute Name"
           attrType                  length=$32      label="Extended Atttibute Type"
           attrIsInherited           length=8        label="Extended Attribute is Inherited"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java items        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/core/type/TypeService");                                                                                  
    declare javaobj results("com/sas/lsaf/macro/core/type/SasTypeAttributeDefinitionsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getTypeExtendedAttributes", "%bquote(&lsaf_typeid)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not retrieve list of extended attributes from %bquote(&lsaf_typeid)" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Returned a list of extended attributes in %bquote(&lsaf_typeid)";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
         results.callStringMethod("getTypeId", row, typeId);
         results.callStringMethod("getTypeName", row, typeName);
         results.callStringMethod("getAttributeId", row, attrId);
         results.callStringMethod("getAttributeName", row, attrName);
         results.callStringMethod("getAttributeType", row, attrType);
         results.callBooleanMethod("attributeIsInherited", row, attrIsInherited);
         output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No extended attributes were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by typeId attrName;
    run;
    options &notesOption;
  %end;

%mend lsaf_gettypeextendedattributes ;
