%* lsaf_gettypeallowablechildren                                                                 *;
%*                                                                                               *;
%* Gets the metadata for the item types that can be created as immediate children of the         *;
%* specified type in the repository. The metadata is stored in a SAS data set.                   *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 2.5.1                                                                                   *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2021-09-22  initial coding                                                                    *;
%* 2019-11-01  update to handle a blank SAS output file.                                         *;
%* 2020-07-13  update for refactor of java service hierarchy.                                    *;
%* 2021-08-17  Remove header documentation                                                       *;
%* 2021-09-07  Update for US11678                                                                *;
%* 2021-10-12  US11739                                                                           *;
%* 2021-10-18  DE17010                                                                           *;
%* 2021-11-15  2.6: Changed message from items to types                                          *;
%*                                                                                               *;

%macro lsaf_gettypeallowablechildren(
  lsaf_typeid=,
  sas_dsname=work.lsafGetTypeAllowableChildren
  ) /des='Get a list of the item types in SAS Life Science Analytics Framework.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetTypeAllowableChildren;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=typeId id name description lsafType
                    allowableChildren capabilities listType
                    );

    attrib returnCode                length=8
           message                   length=$200
           logMsgPrefix              length=$70
           logMsg                    length=$300
           rowCount                  length=8
           logMsgPrimary             length=$70

           listType                  length=$32      label="LSAF List Type"           
           typeId                    length=$2048    label="Type ID"
           id                        length=$2048    label="Child Type ID"
           name                      length=$256     label="Type Name"
           lsafType                  length=$32      label="LSAF Type"
           description               length=$512     label="Description"
           allowableChildren         length=$2048    label="Allowable Children Type"
           capabilities              length=$2048    label="Capabilities"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java items        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/core/type/TypeService");                                                                                  
    declare javaobj results("com/sas/lsaf/macro/core/type/SasTypesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getAllowableChildTypes", "%bquote(&lsaf_typeid)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not retrieve list of types for %bquote(&lsaf_typeid)" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Returned a list of types for %bquote(&lsaf_typeid)";
      results.callIntMethod("getSize", rowCount);
      results.callStringMethod("getListType", listType);                                                             
      typeId="&lsaf_typeid";
      do row=0 to rowCount - 1;
        results.callStringMethod("getId", row, id);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getDescription", row, description);
        results.callStringMethod("getLsafType", row, lsafType);
        results.callStringMethod("getAllowableChildTypes", row, allowableChildren);
        results.callStringMethod("getCapabilities", row, capabilities);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No allowable children were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by id;
    run;
    options &notesOption;
  %end;

%mend lsaf_gettypeallowablechildren ;
