%* lsaf_getsystemconstants                                                                         *;
%*                                                                                                 *;
%* Gets a list of the SAS Life Science Analytics Framework constants that are used as a parameter, *;
%* data set, or return value. The metadata is stored in a SAS data set.                            *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 2.1                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2019-01-24  initial code                                                                        *;
%* 2019-10-02  added macro api version release to data set                                         *;
%* 2019-11-01  update to handle a blank SAS output file.                                           *;
%* 2020-07-13  update for refactor of java service hierarchy.                                      *;
%* 2021-08-09  refactor for surfacing as external macro in 2.5.1                                   *;
%* 2021-08-17  Remove header documentation                                                         *;
%* 2021-10-12  US11739                                                                             *;
%*                                                                                                 *;

%macro lsaf_getsystemconstants(
  lsaf_name=,
  sas_dsname=work.lsafGetSystemConstants
  ) /des='Get a list of the system constants';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetSystemConstants;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the data step to call the java code  *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep= macroApiVersion source name isDefaultValue value details);

    attrib returnCode              length=8
           message                 length=$200
           logMsgPrefix            length=$70
           logMsgPrimary           length=$70
           logMsg                  length=$300
           rowCount                length=8

           macroApiVersion         length=$30    label="Macro API Version"
           source                  length=$150   label="Source"
           name                    length=$64    label="Name"
           isDefaultValue          length=8      label="Is Default Value"
           value                   length=$100   label="Value"
           details                 length=$264   label="Details"
	;

    rowCount=0;                        

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/core/system/SystemService");
    declare javaobj results("com/sas/lsaf/macro/core/system/SasSystemConstantsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getSystemConstants", "%bquote(&lsaf_name)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of constants.";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of constants was returned.";
	  results.callStringMethod("getMacroRelease", macroApiVersion);
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getSource", row, source);
        results.callStringMethod("getName", row, name);
		results.callStringMethod("getValue", row, value);
        results.callBooleanMethod("isDefaultValue", row, isDefaultValue);
		results.callStringMethod("getDetails", row, details);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0)
      then put logMsgPrefix " No constants were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by name source;
    run;
    options &notesOption;
  %end;

%mend lsaf_getsystemconstants;
