%* lsaf_getsyncinfo                                                                                *;
%*                                                                                                 *;
%* Returns a SAS data set that contains synchronization information for a file that is in the      *;
%* repository.                                                                                     *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 1.5                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2016-10-28  initial coding.                                                                     *;
%* 2020-09-04  re-implemented at 2.4.                                                              *;
%* 2021-08-13  Remove header documentation                                                         *;
%*                                                                                                 *;

%macro lsaf_getsyncinfo(
   lsaf_path=,
   sas_dsname=work.lsafGetSyncInfo
   ) /des='Get the synchronization information for a file.';
   
   %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetSyncInfo;

   %****************************************;
   %* Initialize macro variables           *;
   %****************************************;
   %lsaf_getProductName();

   %global _lsafMsg_
           _lsafRC_
   ;

   %let _lsafRC_ =%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

   %**********************************************;
   %* Start the datastep to call the java code   *;
   %**********************************************;
   data &sas_dsname (label="Created by &SYSMACRONAME"
                     keep=path name value description
   );

   attrib rowCount        length=8
          returnCode      length=8
          message         length=$200
          logMsgPrefix    length=$70
          logMsg          length=$300
		  path            length=$2048    label="Item Path"
          name            length=$32      label="Sync Information Name"
          value	          length=$2048    label="Sync Information Value"
          description     length=$200     label="Sync Information Description"
   ;

   rowCount=0;

   %***********************************;
   %* Declare the java objects        *;
   %***********************************;
   declare javaobj srv("com/sas/lsaf/macro/content/repository/RepositoryService") ;
   declare javaobj results("com/sas/lsaf/macro/content/repository/SasSyncPropertiesResult");

   %***********************************;
   %* Attempt the operation           *;
   %***********************************;
   srv.callStaticVoidMethod("getSynchInfo", "%bquote(&lsaf_path)", results) ;

   %***********************************;
   %* Retrieve the results            *;
   %***********************************;
   results.callIntMethod("getReturnCode", returnCode);
   results.callStringMethod("getReturnMessage", message);

   %***********************************;
   %* Process the results             *;
   %***********************************;
   logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
   if (returnCode ne 0) then
   do;
     logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
     logMsg=" Cannot retrieve synchronization information for: &lsaf_path" ;
   end;
   else
   do;
     logMsgPrimary=logMsgPrefix;
     logMsg=" Returned a list of workspace objects in %bquote(&lsaf_path)";
     results.callIntMethod("getSize", rowCount);
     do row=0 to rowCount - 1;
       path="&lsaf_path";
       results.callStringMethod("getPropertyName", row, name);
       results.callStringMethod("getPropertyValue", row, value);
       results.callStringMethod("getPropertyDescription", row, description);
       output;
     end;
   end;

   %***********************************;
   %* Print messages in log           *;
   %***********************************;
   put;
   put logMsgPrimary " " logMsg;
   put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
   if (rowCount=0) then
     put logMsgPrefix " No synchronization information was found.";
   else
     put logMsgPrefix " Synchronization information written to data set &sas_dsname";
   put logMsgPrefix " _lsafMsg_= " message ;
   put logMsgPrefix " _lsafRC_= " returnCode ;
   put;

   %***********************************;
   %* Set the macro variables         *;
   %***********************************;
   call symputx("_lsafRC_", returnCode);
   call symputx("_lsafMsg_", message);

   run;

%mend lsaf_getsyncinfo;
