%* lsaf_getsubscriptions                                                                         *;
%*                                                                                               *;
%* Gets the metadata for all subscriptions for the current user. The metadata is stored in a     *;
%* SAS data set.                                                                                 *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 1.4                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2014-03-11  initial coding                                                                    *;
%* 2016-08-26  rebrand as LSAF                                                                   *;
%* 2019-02-15  update to 2.1                                                                     *;
%* 2019-11-01  update to handle a blank SAS output file.                                         *;
%* 2020-07-13  update for refactor of java service hierarchy.                                    *;
%* 2021-08-17  Remove header documentation                                                       *;
%* 2021-10-13  US11739                                                                           *;
%* 2021-11-15  US11761 Added path parameter                                                      *;
%*                                                                                               *;

%macro lsaf_getsubscriptions(
  lsaf_path=,
  sas_dsname=work.lsafGetSubscriptions
  ) /des='Get a list of subscriptions for the current user';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetSubscriptions;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_
         notesOption
  ;

  %let _lsafRowCount_=0;

  %**********************************************;
  %*  Start the datastep to call the java code  *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME" keep=event path);
    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$500
		   pathLogMsg      length=$20
           rowCount        length=8

           event           length=$64       label="Event Name"
           path            length=$2048     label="Path"
     ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/messaging/subscription/SubscriptionService");
    declare javaobj results("com/sas/lsaf/macro/messaging/subscription/SasSubscriptionsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getSubscriptionsForCurrentUser", "%bquote(&lsaf_path)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
	if ("&lsaf_path" ne "") then 
	do;
	   pathLogMsg = " for specified path";
	end;
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve subscriptions" || trim(pathLogMsg) ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Subscriptions were returned for the current user" || trim(pathLogMsg) ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getEvent", row, event);
        results.callStringMethod("getPath", row, path);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No subscriptions were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by event path;
    run;
    options &notesOption;
  %end;

%mend lsaf_getsubscriptions;
