%* lsaf_getstudytlfsinfo                                                                   *;
%*                                                                                         *;
%* Gets the Study TLFs information.                                                        *;
%*                                                                                         *;
%* macrotype LSAF                                                                          *;
%* since 2.5                                                                               *;
%* exposure external                                                                       *;
%*                                                                                         *;
%* History                                                                                 *;
%* 2020-03-18  initial coding.                                                             *;
%* 2021-08-17  Remove header documentation                                                 *;
%* 2021-10-12  US11739                                                                     *;
%*                                                                                         *;

%macro lsaf_getstudytlfsinfo(
  lsaf_studyId=, 
  lsaf_basestandardname=,
  lsaf_modelId=,
  lsaf_sectionId=, 
  lsaf_recurse=, 
  sas_dsname=work.lsafgetstudytlfsinfo
  ) /des='Gets the Study TLFs information.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetstudytlfsinfo;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname. (keep=studyId baseStandardName modelId sectionId tlfId tlfIdentifier tlfName tlfType tlfAutoflowEnabled);
    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300
           rowCount        length=8

           studyId            length=$70   label="Study ID" 
           baseStandardName   length=$15   label="Base Data Standard Name" 
           modelId            length=$64   label="Model ID"
           sectionId          length=$64   label="Section ID"
           tlfId              length=$64   label="TLF ID" 
           tlfIdentifier      length=$256  label="TLF Identifier"
           tlfName            length=$200  label="TLF Name"
           tlfType            length=$15   label="TLF Type"
           tlfAutoflowEnabled length=8     label="TLF Autoflow Enabled"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/clinical/study/tlf/StudyTlfService");
    declare javaobj results("com/sas/lsaf/macro/clinical/study/tlf/SasStudyTlfsInfoResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getTlfsInfo", "%bquote(&lsaf_studyId)", "%bquote(&lsaf_baseStandardName)", "%bquote(&lsaf_modelId)",
                             "%bquote(&lsaf_sectionId)", "%bquote(&lsaf_recurse)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve study TLFs for the study." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Study TLFs were returned for the study." ;
      results.callIntMethod("getSize", rowCount);
      studyId = "&lsaf_studyId";
      baseStandardName = "&lsaf_baseStandardName";
       modelId = "&lsaf_modelId";
      do row=0 to rowCount - 1;
        results.callStringMethod("getParentId", row, sectionId);
		results.callStringMethod("getId", row, tlfId);
        results.callStringMethod("getIdentifier", row, tlfIdentifier);
        results.callStringMethod("getName", row, tlfName);
        results.callStringMethod("getType", row, tlfType);
        results.callBooleanMethod("autoflowEnabled", row, tlfautoflowenabled);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No study TLFs were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);
  run;

  %***********************************;
  %* Process the data set            *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname. sortseq=linguistic(strength=primary);
      by studyId basestandardname modelid sectionId tlfIdentifier;
    run;
    options &notesOption;
  %end;

%mend lsaf_getstudytlfsinfo;
