%* lsaf_getstudytlfresultdatasets                                                           *;
%*                                                                                          *;
%* Gets a list of TLF analysis result data sets.                                            *;
%*                                                                                          *;
%* macrotype LSAF                                                                           *;
%* since 2.5                                                                                *;
%* exposure external                                                                        *;
%*                                                                                          *;
%* History                                                                                  *;
%* 2020-05-19  initial coding.                                                              *;
%* 2020-06-29  rename lsaf_resultId parameter to lsaf_resultIdentifier.                     *;
%* 2021-08-17  Remove header documentation                                                  *;
%*                                                                                          *;

%macro lsaf_getstudytlfresultdatasets(
  lsaf_studyId=, 
  lsaf_basestandardname=,
  lsaf_modelId=,
  lsaf_resultIdentifier=,
  sas_dsname=work.lsafgetstudytlfresultdatasets
  ) /des='Gets a list of TLF analysis result data sets.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetstudytlfresultdatasets;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname. (keep=studyId baseStandardName modelId resultIdentifier name analysisColumns whereClause);
    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300
           rowCount        length=8

           studyId            length=$70    label="Study identifier of the analysis result" 
           baseStandardName   length=$15    label="Base standard of the analysis result"
           modelId            length=$64    label="Model identifier of the analysis result"
           resultIdentifier   length=$64    label="Result identifier of the analysis result"
           name               length=$64    label="Name of the analysis data set"
           analysisColumns    length=$2048  label="Analysis columns of the data set"
           whereClause        length=$256   label="Where clause of the data set"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/clinical/study/tlf/StudyTlfService");
    declare javaobj results("com/sas/lsaf/macro/clinical/tlf/SasTlfAnalysisDatasetsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getTlfAnalysisDatasets", "%bquote(&lsaf_studyId)", "%bquote(&lsaf_baseStandardName)", 
                             "%bquote(&lsaf_modelId)", "%bquote(&lsaf_resultIdentifier)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve TLF analysis result data sets for the study." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Study TLF analysis result data sets were returned for the study." ;
      results.callIntMethod("getSize", rowCount);
      studyId="&lsaf_studyId";
      resultIdentifier="&lsaf_resultIdentifier";
      modelId="&lsaf_modelId";
      baseStandardName="&lsaf_basestandardname";
      do row=0 to rowCount - 1;
        results.callStringMethod("getTitle", row, name);
        results.callStringMethod("getVariables", row, analysisColumns);
        results.callStringMethod("getWhereClause", row, whereClause);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No study TLF analysis result data sets were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);
  run;

%mend lsaf_getstudytlfresultdatasets;
