%* lsaf_getstudytlfanalysisresults                                                          *;
%*                                                                                          *;
%* Gets a list of TLF analysis results.                                                     *;
%*                                                                                          *;
%* macrotype LSAF                                                                           *;
%* since 2.5                                                                                *;
%* exposure external                                                                        *;
%*                                                                                          *;
%* History                                                                                  *;
%* 2020-05-19  initial coding.                                                              *;
%* 2021-08-17  Remove header documentation                                                  *;
%*                                                                                          *;

%macro lsaf_getstudytlfanalysisresults(
  lsaf_studyId=, 
  lsaf_basestandardname=,
  lsaf_modelId=,
  lsaf_tlfIdentifier=,
  sas_dsname=work.lsafgetstudytlfanalysisresults
  ) /des='Gets a list of TLF analysis results.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetstudytlfanalysisresults;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname. (keep=studyId baseStandardName modelId tlfIdentifier resultIdentifier description reason purpose 
                          analysisParameter documentation code codeContext tableJoinComment order dataSetCount);
    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300
           rowCount        length=8

           studyId            length=$70     label="Study identifier of the analysis result"
           baseStandardName   length=$15     label="Base standard of the analysis result"
           modelId            length=$64     label="Model identifier of the analysis result"
           tlfIdentifier      length=$64     label="TLF identifier"
           resultIdentifier   length=$64     label="Unique identifier of the analysis result"
           description        length=$2048   label="Description of the analysis result"
           reason             length=$2048   label="Reason for the analysis result"
           purpose            length=$2048   label="Purpose of the analysis result"
           analysisParameter  length=$2048   label="Analysis parameter of the analysis result"
           documentation      length=$2048   label="Documentation of the analysis result"
           code               length=$32767  label="Code for the analysis result"
           codeContext        length=$64     label="Code context of the analysis result"
           tableJoinComment   length=$2048   label="Comment for a table join of the analysis result"
           order              length=8.0     label="Order number of the analysis result"
           dataSetCount       length=8.0     label="Data set count for the analysis result"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/clinical/study/tlf/StudyTlfService");
    declare javaobj results("com/sas/lsaf/macro/clinical/tlf/SasTlfAnalysesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getTlfAnalysisResults", "%bquote(&lsaf_studyId)", "%bquote(&lsaf_baseStandardName)", 
                             "%bquote(&lsaf_modelId)", "%bquote(&lsaf_tlfIdentifier)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve TLF analysis results for the study.";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Study TLF analysis results were returned for the study.";
      results.callIntMethod("getSize", rowCount);
      studyId="&lsaf_studyId";
      tlfIdentifier="&lsaf_tlfIdentifier";
      modelId="&lsaf_modelId";
      baseStandardName="&lsaf_basestandardname";
      do row=0 to rowCount - 1;
        results.callStringMethod("getDescription", row, description);
        results.callStringMethod("getReason", row, reason);
        results.callStringMethod("getPurpose", row, purpose);
        results.callStringMethod("getAnalysisParameter", row, analysisParameter);
        results.callStringMethod("getIdentifier", row, resultIdentifier);
        results.callStringMethod("getDocumentation", row, documentation);
        results.callStringMethod("getCode", row, code);
        results.callStringMethod("getCodeContext", row, codeContext);
        results.callStringMethod("getTableJoinComment", row, tableJoinComment);
        results.callIntMethod("getOrder", row, order);
        results.callIntMethod("getDatasetCount", row, dataSetCount);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No study TLF analysis results were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);
  run;

%mend lsaf_getstudytlfanalysisresults;
