%* lsaf_getstudydictionaries                                                                        *;
%*                                                                                                  *;
%* Gets the metadata for all of the controlled terminology dictionaries that are associated with a  *;
%* study. The metadata is stored in a SAS data set.                                                 *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.2                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2019-08-14  initial coding                                                                       *;
%* 2019-11-01  update to handle a blank SAS output file.                                            *;
%* 2021-08-17  Remove header documentation                                                          *;
%* 2021-08-25  DE16771                                                                              *;
%*                                                                                                  *;

%macro lsaf_getstudydictionaries(
  lsaf_studyid=,
  sas_dsname=work.lsafgetstudydictionaries
  ) /des='Get a list of all controlled terminology dictionaries for a SAS Life Science Analytics Framework study.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetstudydictionaries;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;


  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep= studyId name id standard description isCheckedOut checkedOutBy checkedOutDate checkedOutDateSASFormat
                          createdDate createdDateSASFormat createdBy state publishedState currentVersion associatedStudiesCount
                          lastModifiedDate lastModifiedDateSASFormat lastModifiedBy releaseDate dictionaryStatus
  );

    attrib returnCode                length=8;
    attrib message                   length=$200;
    attrib logMsgPrefix              length=$70;
    attrib logMsgPrimary             length=$70;
    attrib logMsg                    length=$300;
    attrib rowCount                  length=8;

    attrib studyId                   length=$64    label="Study ID";
    attrib name                      length=$128   label="Name";
    attrib id                        length=$64    label="ID";
    attrib standard                  length=$64    label="Standard";
    attrib description               length=$64    label="Description";
    attrib isCheckedOut              length=8      label="Is Checked Out";
    attrib checkedOutBy              length=$64    label="Checked Out By";    
    attrib checkedOutDate            length=$64    label="Checked Out Date";
    attrib checkedOutDateSASFormat   length=$64    label="Checked Out Date (SAS Format)";
    attrib createdDate               length=$64    label="Date Created";
    attrib createdDateSASFormat      length=$64    label="Date Created (SAS Format)";
    attrib createdBy                 Length=$64    Label="Created by";
    attrib state                     length=$64    label="State";
    attrib publishedState            length=$64    label="Published State";
    attrib currentVersion            length=$32    label="Current Version";
    attrib associatedStudiesCount    length=8      label="Associated Studies Count";
    attrib lastModifiedDate          length=$64    label="Date Last Modified";
    attrib lastModifiedDateSASFormat length=$64    label="Date Last Modified (SAS Format)";
    attrib lastModifiedBy            length=$64    label="Last Modified By";    
    attrib releaseDate               length=$64    label="Date of Release";
    attrib dictionaryStatus          length=$64    label="Dictionary Status";

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/clinical/study/StudyService");
    declare javaobj results("com/sas/lsaf/macro/clinical/dictionary/SasDictionariesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getStudyDictionaries", "%bquote(&lsaf_studyid)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of controlled terminology dictionaries.";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of controlled terminology dictionaries was returned.";
      results.callIntMethod("getSize", rowCount);
      studyId=kstrip("&lsaf_studyid");
      do row=0 to rowCount - 1;
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getId", row, id);
        results.callStringMethod("getStandard", row, standard);
        results.callStringMethod("getDescription", row, description);
        results.callIntMethod("isCheckedOut", row, isCheckedOut);
        results.callStringMethod("getCheckedOutBy", row, checkedOutBy);
        results.callStringMethod("getCheckedOutDate", row, checkedOutDate);
        results.callStringMethod("getCheckedOutDataSasFormat", row, checkedOutDateSASFormat);
        results.callStringMethod("getCreated", row, createdDate);
        results.callStringMethod("getCreatedSasFormat", row, createdDateSasFormat);
        results.callStringMethod("getCreatedBy", row, createdBy);
        results.callStringMethod("getState", row, state);
        results.callStringMethod("getPublishedState", row, publishedState);
        results.callStringMethod("getCurrentVersion", row, currentVersion);
        results.callLongMethod("getAssociatedStudiesCount", row, associatedStudiesCount);
        results.callStringMethod("getLastModified", row, lastModifiedDate);
        results.callStringMethod("getLastModifiedSasFormat", row, lastModifiedDateSASFormat);
        results.callStringMethod("getLastModifiedBy", row, lastModifiedBy);
        results.callStringMethod("getReleaseDate", row, releaseDate);
        results.callStringMethod("getDictionaryStatus", row, dictionaryStatus);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No controlled terminology dictionaries were found for study item ID &lsaf_studyid..";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);

  run;

%mend lsaf_getstudydictionaries;