%* lsaf_getstdytlfstitlesandfooters                   *;
%*                                                    *;
%* macrotype LSAF                                     *;
%* since 2.5                                          *;
%* exposure external                                  *;
%*                                                    *;
%* History                                            *;
%* 2020-03-24  initial coding.                        *;
%* 2021-08-17  Remove header documentation            *;
%* 2023-03-09  DE17654 - SAS Notes option updates     *;
%*                                                    *;

%macro lsaf_getstdytlfstitlesandfooters(
  lsaf_studyId=, 
  lsaf_basestandardname=,
  lsaf_modelId=,
  lsaf_tlfIdentifier=, 
  sas_dsname=work.lsafgetstdytlfstitlesandfooters
  ) /des='Gets the TITLE and FOOTNOTE properties from a study TLF.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetstdytlfstitlesandfooters;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsaftempStudyTlfDS_ 
          macroName
          notesOption
  ;
  %let _lsaftempStudyTlfDS_ = %str(_lsaftempStudyTlfs_tempDS_);

  %let macroName=%sysfunc(lowcase(&SYSMACRONAME));

  %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
  %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );
  %let logMsg=;
  %let rowCount=0;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %let notesOption=%sysfunc(getoption(notes));

  %*******************************************************************;
  %* Clean up from previous runs.                                    *;
  %*******************************************************************;

  %if (%sysfunc(exist(&_lsaftempStudyTlfDS_.)) > 0) %then
  %do;
    %put NOTE: &macroName.: Deleting temporary data set &_lsaftempStudyTlfDS_ before processing.;
    options nonotes;
	proc datasets library=work nolist;
      delete &_lsaftempStudyTlfDS_.;
    quit; run;
	options &notesOption.;
  %end;

  %*******************************************************************;
  %* Validate that the intended output data set is a valid name.     *;
  %*******************************************************************;

  %lsaf_validate_sasdataset_name(callingMacro=&macroName., sas_dsn=&sas_dsname., dsType=Output);

  %if (&_lsafRC_. ne 0) %then
  %do;
    %GOTO PRINT_LOG_MESSAGES;
  %end;

  %*******************************************************************;
  %* Other macros create an empty data set when the macro fails.     *;
  %* To be consistent with that, am creating an empty data set here. *;
  %*******************************************************************;
  option nonotes;
  proc sql;
   create table &sas_dsname.
       (studyId char(32),
        baseStandardName char(15),
        modelId char(64),
        tlfIdentifier char(256));
  quit;

  options &notesOption.;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &_lsaftempStudyTlfDS_.(keep=studyId baseStandardName modelId tlfIdentifier sasName description dataType logicalType dataLength ordering value);
    attrib returnCode      length=8
           message         length=$200
           targetedMsg     length=$25
           rowCount        length=8

           studyId            length=$70  label="Study ID" 
           baseStandardName   length=$15  label="Base Data Standard Name" 
           modelId            length=$64  label="Model ID"
           tlfIdentifier      length=$256 label="TLF Identifier"
           sasName            length=$32
           description        length=$200
           ordering           length=8
           dataType           length=$15
           logicalType        length=$15
           dataLength         length=8
           value              length=$3000  label=
    ;

    rowCount=0;

    targetedMsg = "for TLF &lsaf_tlfIdentifier.";
    if (kstrip("&lsaf_tlfIdentifier.") eq "") then
       targetedMsg="TLFs for the study.";

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/clinical/study/tlf/StudyTlfService") ;
    declare javaobj results("com/sas/lsaf/macro/clinical/study/tlf/SasStudyTlfPropertiesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getTlfTitlesAndFootnotes", "%bquote(&lsaf_studyId)", "%bquote(&lsaf_baseStandardName)", "%bquote(&lsaf_modelId)",
                             "%bquote(&lsaf_tlfIdentifier)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    if (returnCode ne 0) then
    do;
      call symput('logMsg', " Cannot retrieve titles and footnotes " || strip(targetedMsg) || ".");
      STOP;
    end;

    call symput('logMsgPrimary', "&logMsgPrefix.");
    results.callIntMethod("getSize", rowCount);

    if (rowCount eq 0) then 
    do;
      call symput('logMsg', "No TLF titles and footnotes were returned for the study.");
    end;
    else 
    do;
      call symput('logMsg', "List of TLF titles and footnotes was returned for the study.");
      call symputx("_lsafRowCount_", rowCount);
    end;

    studyId = "&lsaf_studyId.";
    baseStandardName = "&lsaf_baseStandardName";
    modelId = "&lsaf_modelId.";
    do row=0 to rowCount - 1;
      results.callStringMethod("getTlfIdentifier", row, tlfIdentifier);
      results.callStringMethod("getSasName", row, sasName);
      results.callStringMethod("getDescription", row, description);
      results.callIntMethod("getOrdering", row, ordering);
      results.callStringMethod("getDataType", row, dataType);
      results.callStringMethod("getLogicalType", row, logicalType);
      results.callIntMethod("getLength", row, dataLength);
      results.callStringMethod("getValue", row, value);
      output;
    end;

  run;

  %**********************************************************;
  %* Stop here if the server call failed.                   *;
  %**********************************************************;

  %if (&_lsafRC_. ne 0) %then
  %do;
    %GOTO PRINT_LOG_MESSAGES;
  %end;

  %**********************************************************;
  %* Process the data set                                   *;
  %**********************************************************;
  option nonotes;
  proc sort data=&_lsaftempStudyTlfDS_.;
    by studyId baseStandardName tlfIdentifier modelId ordering;
  run;

  data _null_;
    attrib setting length=$10 attribCode length=$5000;
    retain attribCode;
    set &_lsaftempStudyTlfDS_. end=eof;
    by studyId baseStandardName modelId tlfIdentifier ordering;
    if (first.tlfIdentifier AND NOT(_n_=1)) then 
    do;
      goto FINISH;
    end;
    if (dataType eq "STRING") then setting = '$' || strip(put(dataLength, best.));
    else if (dataType eq "BOOLEAN") then setting = '$3';
    else if (logicalType eq "DATE") then setting = '$22';
    attribCode = strip( catx(' ', attribCode,strip(sasName), cats('length=' || strip(setting))) );
    if (eof) then 
    do;
      goto FINISH;
    end;
    return;
    FINISH:
    call symput('_lsafattribcode_', strip(attribCode));
    STOP;
  run;

  proc transpose data=&_lsaftempStudyTlfDS_. (drop=ordering) out=&sas_dsname. (drop=_name_);
    by studyId baseStandardName modelId tlfIdentifier;
    id sasName;
    idlabel description; 
    var value;
  run;

  %***********************************;
  %* Clean up artifacts.             *;
  %***********************************;

  %if (%sysfunc(exist(&_lsaftempStudyTlfDS_.))>0) %then
  %do;
    proc datasets library=work nolist;
      delete &_lsaftempStudyTlfDS_.;
    quit; 
    run;
  %end;

  %***********************************;
  %* Print log messages.             *;
  %***********************************;

  %let mydataID=%sysfunc(OPEN(&sas_dsname.,IN));
  %let rowCount=%sysfunc(ATTRN(&mydataID,NOBS));
  %let RC=%sysfunc(CLOSE(&mydataID));

  %if (&rowCount eq 1) %then 
  %do;
    data &sas_dsname;
      set &sas_dsname;
      if (strip(studyId) ne "") AND (strip(baseStandardName) ne "") AND
         (strip(modelId) ne "") AND (strip(tlfIdentifier) ne "");
    run;
  %end;

  %PRINT_LOG_MESSAGES:

  options &notesOption;
  %put;
  %put &logMsgPrimary &logMsg ;
  %put &logMsgPrefix &macroName ;
  %put &logMsgPrefix &rowCount records written to data set &sas_dsname;
  %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;      
  %put &logMsgPrefix _lsafMsg_= &_lsafMsg_ ;
  %put;

%mend lsaf_getstdytlfstitlesandfooters;
