%* lsaf_getstandardstudies                                                                          *;
%*                                                                                                  *;
%* Gets the metadata for all of the studies that are associated with a data standard. The           *;
%* metadata is stored in a SAS data set.                                                            *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.2                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2019-06-12  initial coding                                                                       *;
%* 2019-11-01  update to handle a blank SAS output file.                                            *;
%* 2021-08-17  Remove header documentation                                                          *;
%* 2021-08-25  DE16770                                                                              *;
%* 2021-10-12  US11739                                                                              *;
%*                                                                                                  *;

%macro lsaf_getstandardstudies(
  lsaf_standardid=,
  sas_dsname=work.lsafgetstandardstudies
  ) /des='Get a list of all studies by standard in the system.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetstandardstudies;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep= id name path typeid);

    attrib returnCode                length=8;
    attrib message                   length=$200;
    attrib logMsgPrefix              length=$70;
    attrib logMsgPrimary             length=$70;
    attrib logMsg                    length=$300;
    attrib rowCount                  length=8;

    attrib id                        length=$64    label="Study ID";
    attrib name                      length=$128   label="Study Name";
    attrib path                      length=$2048  label="Study Path";
    attrib typeid                    length=$64    label="Context Type ID";   

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/clinical/study/StudyService");
    declare javaobj results("com/sas/lsaf/macro/clinical/study/SasStudiesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getStandardStudies", "%bquote(&lsaf_standardid)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of studies.";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of studies was returned.";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getId", row, id);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getPath", row, path);
        results.callStringMethod("getTypeId", row, typeid);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No studies were found for standard ID &lsaf_standardid..";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by id;
    run;
    options &notesOption;
  %end;

%mend lsaf_getstandardstudies;
