%* lsaf_getstandardsbystatus                                                                        *;
%*                                                                                                  *;
%* Gets the metadata for all of the data standards by the specified status. The metadata is stored  *;
%* in a SAS data set.                                                                               *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.2                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2019-06-07  initial coding                                                                       *;
%* 2019-11-01  update to handle a blank SAS output file.                                            *;
%* 2021-08-16  Remove header documentation                                                          *;
%* 2021-10-12  US11739                                                                              *;
%*                                                                                                  *;

%macro lsaf_getstandardsbystatus(
  lsaf_status=,
  sas_dsname=work.lsafgetstandardsbystatus
  ) /des='Get a list of all standards in the system by status.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetstandardsbystatus;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep= id name baseStandardType baseStandardName baseStandardVersion modelId description
                          isCheckedOut checkedOutBy checkedOutDate checkedOutDateSASFormat createdDate createdDateSASFormat
                          createdBy state publishedState currentVersion associatedStudiesCount lastModifiedDate 
                          lastModifiedDateSASFormat lastModifiedBy standardStatus
  );

    attrib returnCode                length=8;
    attrib message                   length=$200;
    attrib logMsgPrefix              length=$70;
    attrib logMsgPrimary             length=$70;
    attrib logMsg                    length=$300;
    attrib rowCount                  length=8;

    attrib id                        length=$64    label="ID";
    attrib name                      length=$128   label="Name";
    attrib baseStandardType          length=$64    label="Base Standard Type";
    attrib baseStandardName          length=$64    label="Base Standard Name";   
    attrib baseStandardVersion       length=$128   label="Base Standard Version";
    attrib modelId                   length=$128   label="Model ID";
    attrib description               length=$64    label="Description";
    attrib isCheckedOut              length=8      label="Is Checked Out";
    attrib checkedOutBy              length=$64    label="Checked Out By";    
    attrib checkedOutDate            length=$64    label="Checked Out Date";
    attrib checkedOutDateSASFormat   length=$64    label="Checked Out Date (SAS Format)";
    attrib createdDate               length=$64    label="Date Created";
    attrib createdDateSASFormat      length=$64    label="Date Created (SAS Format)";
    attrib createdBy                 Length=$64    Label="Created by";
    attrib state                     length=$64    label="State";
    attrib publishedState            length=$64    label="Published State";
    attrib currentVersion            length=$32    label="Current Version";
    attrib associatedStudiesCount    length=8      label="Associated Studies Count";
    attrib lastModifiedDate          length=$64    label="Date Last Modified";
    attrib lastModifiedDateSASFormat length=$64    label="Date Last Modified (SAS Format)";
    attrib lastModifiedBy            length=$64    label="Last Modified By";    
    attrib standardStatus            length=$64    label="Standard Status";

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/clinical/standards/StandardsService");
    declare javaobj results("com/sas/lsaf/macro/clinical/standards/SasStandardsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getStandardsByStatus", "%bquote(&lsaf_status)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of standards.";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of standards was returned.";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getId", row, id);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getBaseStandardType", row, BaseStandardType);
        results.callStringMethod("getBaseStandardName", row, BaseStandardName);
        results.callStringMethod("getBaseStandardVersion", row, BaseStandardVersion);
        results.callStringMethod("getModelId", row, modelId);
        results.callStringMethod("getDescription", row, description);
        results.callIntMethod("isCheckedOut", row, isCheckedOut);
        results.callStringMethod("getCheckedOutBy", row, checkedOutBy);
        results.callStringMethod("getCheckedOutDate", row, checkedOutDate);
        results.callStringMethod("getCheckedOutDataSasFormat", row, checkedOutDateSASFormat);
        results.callStringMethod("getCreated", row, createdDate);
        results.callStringMethod("getCreatedSasFormat", row, createdDateSasFormat);
        results.callStringMethod("getCreatedBy", row, createdBy);
        results.callStringMethod("getState", row, state);
        results.callStringMethod("getPublishedState", row, publishedState);
        results.callStringMethod("getCurrentVersion", row, currentVersion);
        results.callLongMethod("getAssociatedStudiesCount", row, associatedStudiesCount);
        results.callStringMethod("getLastModified", row, lastModifiedDate);
        results.callStringMethod("getLastModifiedSasFormat", row, lastModifiedDateSASFormat);
        results.callStringMethod("getLastModifiedBy", row, lastModifiedBy);
        results.callStringMethod("getStandardStatus", row, standardStatus);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount<=0) 
      then put logMsgPrefix " No standards were found for status &lsaf_status..";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by name;
    run;
   options &notesOption;
  %end;

%mend lsaf_getstandardsbystatus;