%* lsaf_getsignatures                                                                               *;
%*                                                                                                  *;
%* Gets the metadata for all of the signatures of a file or a specific version of the file.         *;
%* The metadata is stored in a SAS data set.                                                        *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 1.4                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2014-02-25  initial coding.                                                                      *;
%* 2016-08-26  rebrand as LSAF                                                                      *;
%* 2019-01-31  update to 2.1                                                                        *;
%* 2019-11-01  update to handle a blank SAS output file.                                            *;
%* 2020-07-13  update for refactor of java service hierarchy.                                       *;
%* 2021-08-16  Remove header documentation                                                          *;
%* 2021-10-12  US11739                                                                              *;
%*                                                                                                  *;

%macro lsaf_getsignatures(
  lsaf_path=,
  lsaf_version=,
  sas_dsname=work.lsafGetSignatures
  ) /des='Get a list of all the signatures of a file in the SAS Life Science Analytics Framework in the repository.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetSignatures;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_
         notesOption
  ;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=path name itemType version userId dateTimeStamp dateSigned reason role comment);

    attrib returnCode                length=8
           message                   length=$200
           logMsgPrefix              length=$70
           logMsgPrimary             length=$70
           logMsg                    length=$1000
           rowCount                  length=8

           path                      length=$2048    label="File Path"
           name                      length=$256     label="File name"
           itemType                  length=$32      label="Item Type"
           version                   length=$32      label="Version Number"

           userId                    length=$64      label="User ID"
           dateTimeStamp             length=$64      label="Date and time of the signature"
           reason                    length=$64      label="Reason for signature"
           role                      length=$512     label="Role associated with signature"
           comment                   length=$512     label="Comment entered at time of signature"

           timeStampTempSAS          length=$64

           dateSigned             format=datetime.   label="Date of Signature (SAS DateTime Format)"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java files        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/content/repository/signature/SignatureService") ;
    declare javaobj results("com/sas/lsaf/macro/content/repository/signature/SasSignaturesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getSignatures", "%bquote(&lsaf_path)", "%bquote(&lsaf_version)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of signatures for &lsaf_path" ;
    end;
    else
    do;
      lsafVersion=strip(symget("lsaf_version"));
      logMsgPrimary=logMsgPrefix;
      if (length(lsafVersion)>0) then
      do;
         logMsg=" (" || strip(lsafVersion) || ")" ;
      end;
      logMsg=" List of signatures were returned for &lsaf_path" || logMsg ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getPath", row, path);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getItemType", itemType);
        results.callStringMethod("getVersion", row, version);
        results.callStringMethod("getUserId", row, userId);

        results.callStringMethod("getTime", row, dateTimeStamp);

        results.callStringMethod("getReason", row, reason);
        results.callStringMethod("getRole", row, role);
        results.callStringMethod("getComment", row, comment);

        results.callStringMethod("getTimeSasFormat", row, timeStampTempSAS);

        dateSigned=input(timeStampTempSAS, datetime.);

        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No signatures were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

   run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by descending dateSigned;
    run;
    options &notesOption;
  %end;

%mend lsaf_getsignatures;
