%* lsaf_getsharedfolderlinks                                                               *;
%*                                                                                         *;
%* Get the shared workspace entries for the specified repository shared folder.            *;
%*                                                                                         *;
%* macrotype LSAF                                                                          *;
%* since 2.7                                                                               *;
%* exposure external                                                                       *;
%*                                                                                         *;
%* History                                                                                 *;
%* 2024-08-09  initial code                                                                *;
%*                                                                                         *;

%macro lsaf_getsharedfolderlinks(
  lsaf_sharedfolderid=,
  sas_dsname=work.lsafGetSharedFolderLinks
  ) /des='Get shared folders links.';

  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetSharedFolderLinks;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=sharedWorkspaceId path memberId isLinked)
  ;
    attrib returnCode          length=8
           message             length=$200
           logMsgPrefix        length=$70
           logMsgPrimary       length=$70
           logMsg              length=$300
           rowCount            length=8

           sharedWorkspaceId   length=$64    label="Shared Workspace ID"
           path                length=$2048  label="Shared Folder Path"
           memberId            length=$256   label="Member ID"
           isLinked            length=8      label="Is Member Linked"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/content/workspace/SharedWorkspaceService") ;
    declare javaobj results("com/sas/lsaf/macro/content/workspace/SasSharedWorkspaceEntriesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getSharedWorkspaceEntries", "%bquote(&lsaf_sharedfolderid)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve shared folder link information for the specified shared folder." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Shared folder link information was returned." ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getSharedWorkspaceId", row, sharedWorkspaceId);
        results.callStringMethod("getPath", row, path);
        results.callStringMethod("getMemberId", row, memberId);
        results.callIntMethod("isLinked", row, isLinked);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No shared folder link information was found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort by name                    *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by memberId;
    run;
    options &notesOption;
  %end;

%mend lsaf_getsharedfolderlinks;
