%* lsaf_getsessionsbyquery                                                                         *;
%*                                                                                                 *;
%* Gets the metadata for all of the sessions that are based on a user-specified query. The         *;
%* metadata is stored in a SAS data set.                                                           *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 2.4                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2020-11-17  initial coding                                                                      *;
%* 2021-08-12  Remove header documentation                                                         *;
%* 2021-10-12  US11739                                                                             *;
%*                                                                                                 *;

%macro lsaf_getsessionsbyquery(
  lsaf_userid=, 
  lsaf_engine=, 
  lsaf_type=, 
  lsaf_state=, 
  lsaf_submissionlabel=, 
  lsaf_startedfromdate=, 
  lsaf_startedtodate=, 
  lsaf_endedfromdate=, 
  lsaf_endedtodate=, 
  lsaf_sessionlimit=,
  sas_dsname=work.lsafgetsessionsbyquery
  ) /des='Get a list of all the sessions in the system as defined by a user query.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetsessionsbyquery;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep= sessionId dateStarted dateStartedSASFormat dateEnded dateEndedSASFormat sessionType sessionState 
                          sessionOwner submissionLabel clientId
    );

    attrib returnCode              length=8;
    attrib message                 length=$200;
    attrib logMsgPrefix            length=$70;
    attrib logMsgPrimary           length=$70;
    attrib logMsg                  length=$300;
    attrib rowCount                length=8;

    attrib sessionOwner            length=$32    label="Session Owner";
    attrib dateStartedSASFormat                  label="Date Session Started (SAS DateTime Format)";
    attrib dateStarted             length=$64    label="Date Session Started";
    attrib dateEndedSASFormat                    label="Date Session Ended (SAS DateTime Format)";
    attrib dateEnded               length=$64    label="Date Session Ended";
    attrib sessionId               length=$64    label="Session ID";
    attrib sessionType             length=$32    label="Session Type";
    attrib sessionState            length=$32    label="Session State";
    attrib submissionLabel         length=$1024  label="Submission Label";
    attrib clientId                length=8      label="Incremental Number Assigned to Session";
    attrib limitReached            length=3;
    ;

    attrib dateStartedSasFormatTmp length=$64    label="SAS Format: USED FOR FORMATTING - will be dropped";
    attrib dateEndedSasFormatTmp   length=$64    label="SAS Format: USED FOR FORMATTING - will be dropped";

    format dateStartedSASFormat dateEndedSASFormat datetime.;

    rowCount=0;
    limitReached=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/execution/session/SessionService") ;
    declare javaobj results("com/sas/lsaf/macro/execution/session/SasSessionsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;   
    srv.callStaticBooleanMethod("search", "%bquote(&lsaf_userid)", "%bquote(&lsaf_engine)", "%bquote(&lsaf_type)", "%bquote(&lsaf_state)", 
      "%bquote(&lsaf_submissionlabel)", "%bquote(&lsaf_startedfromdate)", "%bquote(&lsaf_startedtodate)", "%bquote(&lsaf_endedfromdate)", 
      "%bquote(&lsaf_endedtodate)", "%bquote(&lsaf_sessionlimit)", results, limitReached);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of queried sessions.";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of all the queried sessions was returned.";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getId", row, sessionId);
        results.callStringMethod("getDateStarted", row, dateStarted);
        results.callStringMethod("getDateStartedSasFormat", row, dateStartedSasFormatTmp);
        results.callStringMethod("getDateEnded", row, dateEnded);
        results.callStringMethod("getDateEndedSasFormat", row, dateEndedSasFormatTmp);
        results.callStringMethod("getType", row, sessionType);
        results.callStringMethod("getState", row, sessionState);
        results.callStringMethod("getOwner", row, sessionOwner);
        results.callStringMethod("getSubmissionLabel", row, submissionLabel);
        results.callIntMethod("getClientId", row, clientId);
        
        dateStartedSasFormat = input(dateStartedSasFormatTmp, datetime.);
        dateEndedSasFormat = input(dateEndedSasFormatTmp, datetime.);

        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No queried sessions were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put logMsgPrefix " Session Limit Reached= " limitReached;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by sessionOwner dateStartedSasFormat dateEndedSasFormat;
    run;
    options &notesOption;
  %end;

%mend lsaf_getsessionsbyquery;
