%* lsaf_getscheduleproperties                                                                              *;
%*                                                                                                         *;
%* Gets the properties for the specified schedule. The properties are stored in a SAS data set.            *;
%*                                                                                                         *;
%* macrotype LSAF                                                                                          *;
%* since 2.3                                                                                               *;
%* exposure external                                                                                       *;
%*                                                                                                         *;
%* History                                                                                                 *;
%* 2019-10-01  initial code                                                                                *;
%* 2020-06-30  update service location                                                                     *;
%* 2021-08-16  Remove header documentation                                                                 *;
%* 2021-10-12  US11739                                                                                     *;
%*                                                                                                         *;

%macro lsaf_getscheduleproperties(
  lsaf_scheduleid=,
  sas_dsname=work.lsafGetScheduleProperties
   )/des='Get the metadata for SAS Life Science Analytics Framework schedule.';

  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetScheduleProperties;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getproductname();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=scheduleId scheduleName name value);
    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300
           rowCount        length=8

           scheduleId      length=$64    label="Schedule ID"
           scheduleName    length=$256   label="Schedule Name"
           name            length=$128   label="Property Name"
           value           length=$512   label="Property Value"
    ;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/execution/schedule/ScheduleService");
    declare javaobj results("com/sas/lsaf/macro/execution/schedule/SasSchedulePropertiesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getScheduleProperties", "%bquote(&lsaf_scheduleid)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not retrieve schedule properties." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Retrieved schedule properties.";
      results.callStringMethod("getScheduleId", scheduleId);
      results.callStringMethod("getScheduleName", scheduleName);
      results.callIntMethod("getSize", rowcount);
      do row=0 to rowCount - 1;
         results.callStringMethod("getPropertyName", row, name);
         results.callStringMethod("getPropertyValue", row, value);
         output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " The schedule properties were not found.";
      else put logMsgPrefix  rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname  sortseq=linguistic(strength=primary);
      by name;
    run;
    options &notesOption;
  %end;

%mend lsaf_getscheduleproperties;
