%* lsaf_getschedulejobparameters                                                                    *;
%*                                                                                                  *;
%* Gets the override parameters for the schedule. The parameters are stored in a SAS data set.      *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.3                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2019-11-20  original creation.                                                                   *;
%* 2020-06-30  update service location                                                              *;
%* 2021-08-16  Remove header documentation                                                          *;
%*                                                                                                  *;

%macro lsaf_getschedulejobparameters(
  lsaf_scheduleid=,
  sas_dsname=work.lsafGetScheduleJobParameters
  ) /des='Get a list of override parameters for the specified schedule.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetScheduleJobParameters;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep= scheduleID scheduleName jobLocation jobPath jobVersion 
                          name type value fileVersion includeSubFolders);

    attrib returnCode        length=8
           message           length=$200
           logMsgPrefix      length=$70
           logMsgPrimary     length=$70
           logMsg            length=$300
           rowCount          length=8
           scheduleID        length=$256   label="Schedule ID"
           scheduleName      length=$256   label="Schedule Name"
           jobLocation       length=$32    label="Job Location"
           jobPath           length=$2048  label="Job Path"
           jobVersion        length=$32    label="Job Version"
           name              length=$256   label="Parameter Name"
           type              length=$32    label="Parameter Type"
           value             length=$2048  label="Override Value"
           fileVersion       length=$32    label="File Version if File Type"
           includeSubFolders length=8      label="Include Container SubFolders"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/execution/schedule/ScheduleService") ;
    declare javaobj results("com/sas/lsaf/macro/execution/schedule/SasScheduleParametersResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getScheduleJobParameters", "%bquote(&lsaf_scheduleid)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve schedule override parameters for &lsaf_scheduleid.." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Schedule override parameters were returned for &lsaf_scheduleid.." ;
      results.callStringMethod("getScheduleId", scheduleID);
      results.callStringMethod("getScheduleName", scheduleName);
      results.callStringMethod("getJobLocation", jobLocation);
      results.callStringMethod("getJobPath", jobPath);
      results.callStringMethod("getJobVersion", jobVersion);

      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getParameterName", row, name);
        results.callStringMethod("getParameterType", row, type);
        results.callStringMethod("getParameterValue", row, value);
        results.callStringMethod("getParameterFileVersion", row, fileVersion);
        results.callIntMethod("getParameterIncludeSubfolders", row, includeSubFolders);       
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No schedule override parameters were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);

  run;
  
%mend lsaf_getschedulejobparameters;
