%* lsaf_getrolesbyprivilege                                                     *;
%*                                                                              *;
%* Gets the metadata for all of the roles at a given context that include the   *;
%* specified privilege.                                                         *;
%*                                                                              *;
%* macrotype LSAF                                                               *;
%* since 2.6                                                                    *;
%* exposure external                                                            *;
%*                                                                              *;
%* History                                                                      *;
%* 2021-11-29  initial coding                                                   *;
%*                                                                              *;

%macro lsaf_getrolesbyprivilege(
  lsaf_path=,
  lsaf_privilege=,
  sas_dsname=
  ) /des='Get a list of roles in a context with the specified privilege.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetrolesbyprivilege;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=privilegeId contextPath name description roleSrcCtxt isInherited
  );
    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300
           rowCount        length=8

           privilegeId     length=$64       label="Privilege Identifier"
           contextPath     length=$2048     label="Context Path" 
           name            length=$256      label="Role Name"
           description     length=$128      label="Role Description"
           roleSrcCtxt     length=$2048     label="Role Context Source Path"
           isInherited     length=8         label="Is role inherited in current context: 0 or 1"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/role/RoleService") ;
    declare javaobj results("com/sas/lsaf/macro/security/role/SasRolesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getRolesByPrivilege", "%bquote(&lsaf_path)", "%bquote(&lsaf_privilege)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve roles with &lsaf_privilege from &lsaf_path" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Roles with &lsaf_privilege were returned from &lsaf_path" ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
	  	privilegeId ="&lsaf_privilege";
        contextPath="&lsaf_path";
        results.callStringMethod("getDefinedPath", row, roleSrcCtxt);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getDescription", row, description);
        results.callIntMethod("isInherited", row, isInherited);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No roles with &lsaf_privilege were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort by name                    *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by name;
    run;
    options &notesOption;
  %end;

%mend lsaf_getrolesbyprivilege;
