%* lsaf_getroles                                                                                  *;
%*                                                                                                *;
%* Gets the metadata for all of the roles that are defined and inherited in a context. The        *;
%* metadata is stored in a SAS data set.                                                          *;
%*                                                                                                *;
%* macrotype LSAF                                                                                 *;
%* since 1.2                                                                                      *;
%* exposure external                                                                              *;
%*                                                                                                *;
%* History                                                                                        *;
%* 2013-08-30  updates for standardization and coding conventions                                 *;
%* 2016-08-26  rebrand as LSAF                                                                    *;
%* 2019-01-25  update to 2.1                                                                      *;
%* 2019-11-01  update to handle a blank SAS output file.                                          *;
%* 2020-12-07  updates for service refactor                                                       *;
%* 2021-08-13  Remove header documentation                                                        *;
%* 2021-08-30  Updated for java refactor: sasRole                                                 *;
%* 2021-10-13  US11739                                                                            *;
%* 2021-11-16  US11755 Updated to accept member parameters                                        *;
%*                                                                                                *;

%macro lsaf_getroles(
  lsaf_path=,
  lsaf_member=,
  lsaf_type=,
  lsaf_group_context=,
  sas_dsname=work.lsafGetRoles
  ) /des='Get a list of roles in a SAS Life Science Analytics Framework context.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetRoles;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=contextPath name description roleSrcCtxt isInherited
  );
    attrib returnCode      length=8;
    attrib message         length=$200;
    attrib logMsgPrefix    length=$70;
    attrib logMsgPrimary   length=$70;
    attrib logMsg          length=$300;
    attrib rowCount        length=8;

    attrib contextPath     length=$2048     label="Context Path" ;
    attrib name            length=$256      label="Role Name";
    attrib description     length=$128      label="Role Description";
    attrib roleSrcCtxt     length=$2048     label="Role Context Source Path";
    attrib isInherited     length=8         label="Is role inherited in current context: 0 or 1";

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/role/RoleService") ;
    declare javaobj member("com/sas/lsaf/macro/security/SasPrincipalDescriptor", 
                           "%bquote(&lsaf_type)", "%bquote(&lsaf_member)", "%bquote(&lsaf_group_context)");
    declare javaobj results("com/sas/lsaf/macro/security/role/SasRolesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getRoles", "%bquote(&lsaf_path)", member, results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
	%if ("&lsaf_member" ne "") %then 
	%do;
	   memberLogMsg = " for specified member ";
	%end;
	%else
	%do;
	   memberLogMsg = "";
	%end;
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve roles" || memberLogMsg || "from &lsaf_path" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Roles were returned" || memberLogMsg || "from &lsaf_path" ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        contextPath="&lsaf_path";
        results.callStringMethod("getDefinedPath", row, roleSrcCtxt);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getDescription", row, description);
        results.callIntMethod("isInherited", row, isInherited);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No roles were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort by name                    *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by name;
    run;
    options &notesOption;
  %end;

%mend lsaf_getroles;
