%* lsaf_getrecyclebinitems                                                                       *;
%*                                                                                               *;
%* Gets the metadata for the items in the SAS Life Science Analytics Framework recycle bin that  *;
%* match the given search criteria. The metadata is stored in a SAS data set.                    *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 1.5                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2017-01-24  original coding.                                                                  *;
%* 2019-02-15  update to 2.1                                                                     *;
%* 2019-02-20  renamed from lsaf_getrbitemsbyquery to lsaf_getrecyclebinitems                    *;
%* 2019-09-10  added includesubfolders parameter                                                 *;
%* 2019-11-01  update to handle a blank SAS output file.                                         *;
%* 2020-07-13  update for refactor of java service hierarchy.                                    *;
%* 2021-08-12  Remove header documentation                                                       *;
%* 2021-08-31  DE16792                                                                           *;
%* 2021-10-12  US11739                                                                           *;
%* 2021-11-22  SAS Macros 2.6: RecycleBin refactor                                               *;
%*                                                                                               *;

%macro lsaf_getrecyclebinitems(
  lsaf_location=,
  lsaf_name=,
  lsaf_deletedby=,
  lsaf_from=,
  lsaf_to=,
  lsaf_includesubfolders=,
  sas_dsname=work.lsafgetrecyclebinitems,
  lsaf_printLimitReachedWarning=1
  ) /des='Get a list of the items in the recycle bin based on given criteria.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetrecyclebinitems;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
          _lsaf_PrintMessage_
          _lsafRBSearchLimitReached_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_
         notesOption
  ;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=path name type identifier size deleted deletedBy dateDeleted lastModified lastModifiedBy dateLastModified);
    attrib returnCode         length=8
           message            length=$200
           logMsgPrefix       length=$70
           logMsgPrimary      length=$200
           logMsg             length=$300
           rowCount           length=8
           limitReached       length=8

           identifier         length=$512                     label="Unique Identifier"
           path               length=$2048                    label="Item Path"
           name               length=$256                     label="Item Name"
           type               length=$32                      label="Item Type"
           size               length=8                        label="Size In Bytes"
           deletedBy          length=$64                      label="Userid that deleted the item"
           deleted            length=$64                      label="Date the item was deleted"
           dateDeleted        length=8      format=datetime.  label="Date the item was deleted (SAS DateTime Format)"
           lastModifiedBy     length=$64                      label="Userid that last modified the item content"
           lastModified       length=$64                      label="Date the item's content was last modified"
           dateLastModified   length=8      format=datetime.  label="Date the item's content was last modified (SAS DateTime Format)"
           tempDateDeleted    length=$64                      label="USED FOR FORMATTING - will be dropped"
           tempDateModified   length=$64                      label="USED FOR FORMATTING - will be dropped"
    ;
    
    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/content/repository/recyclebin/RecycleBinService") ;
    declare javaobj results("com/sas/lsaf/macro/content/repository/recyclebin/SasRecycleBinResult");
    declare javaobj query("com/sas/lsaf/macro/content/repository/recyclebin/SasRecycleBinQuery", 
                                  "%bquote(&lsaf_location)", "%bquote(&lsaf_name)", 
                                  "%bquote(&lsaf_deletedby)", "%bquote(&lsaf_from)", "%bquote(&lsaf_to)", 
                                  "%bquote(&lsaf_includesubfolders)"); 

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getRecycleBinItemsByQuery", query,  results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);
    results.callBooleanMethod("isLimitReached", limitReached);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not retrieve list of items in the recycle bin.";
    end;
    else
    do;
      results.callIntMethod("getSize", rowCount);
      if limitReached then
      do;
        %lsaf_printwarningmessage(lsaf_printmessage=&lsaf_printlimitreachedwarning);
        logMsgPrimary="&_lsaf_PrintMessage_";
        logMsg=" Recycle bin items were found matching the given criteria.";
      end;
      else 
      do;
        logMsgPrimary=logMsgPrefix;
        if (rowCount=0) 
          then logMsg = " No recycle bin items were found matching the given criteria.";
          else logMsg = " Recycle bin items were found matching the given criteria.";
      end;

      do row=0 to rowCount - 1;
        results.callStringMethod("getPath", row, path);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getType", row, type);
        results.callStringMethod("getId", row, identifier);
        results.callLongMethod("getItemSize", row, size);
        results.callStringMethod("getDeletedBy", row, deletedBy);
        results.callStringMethod("getDeleted", row, deleted);
        results.callStringMethod("getDeletedSasFormat", row, tempDateDeleted);
        dateDeleted=input(tempDateDeleted, datetime.);
        results.callStringMethod("getLastModifiedBy", row, lastModifiedBy);
        results.callStringMethod("getLastModified", row, lastModified);
        results.callStringMethod("getLastModifiedSasFormat", row, tempDateModified);
        dateLastModified=input(tempDateModified, datetime.);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put logMsgPrefix " _lsafRBSearchLimitReached_= " limitReached;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);
    call symputx("_lsafRBSearchLimitReached_", limitReached);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname out=&sas_dsname sortseq=linguistic(strength=primary);
      by path;
    run;
    options &notesOption;
  %end;

%mend lsaf_getrecyclebinitems;
