%* lsaf_getquerycolumns                                                                            *;
%*                                                                                                 *;
%* Gets the metadata for the columns that are applicable for the specified query type.             *;
%* The metadata is stored in a SAS data set.                                                       *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 2.3                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2020-02-25  initial coding                                                                      *;
%* 2021-08-12  Remove header documentation                                                         *;
%*                                                                                                 *;

%macro lsaf_getquerycolumns(
  lsaf_querytype=,
  sas_dsname=work.lsafgetquerycolumns
  ) /des='Get a list of the metadata for the query columns.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetquerycolumns;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=queryType columnQueryClass columnName columnType 
    );

    attrib returnCode              length=8;
    attrib message                 length=$300;
    attrib logMsgPrefix            length=$70;
    attrib logMsgPrimary           length=$70;
    attrib logMsg                  length=$300;
    attrib rowCount                length=8;

    attrib queryType               length=$32    label="Query Type";
    attrib columnQueryClass        length=$32    label="Column Query Class";
    attrib columnName              length=$32    label="Column Name";
    attrib columnType              length=$32    label="Column Type";

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/query/QueryService");
    declare javaobj results("com/sas/lsaf/macro/query/SasQueryColumnsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getQueryColumns", "%bquote(&lsaf_querytype)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of query columns for &lsaf_querytype..";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of all the query columns was returned.";
      results.callIntMethod("getSize", rowCount);
      results.callStringMethod("getQueryType", queryType);
      do row=0 to rowCount - 1;
        results.callStringMethod("getColumnQueryClass", row, columnQueryClass);
        results.callStringMethod("getColumnName", row, columnName);
        results.callStringMethod("getColumnType", row, columnType);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No query columns were found for &lsaf_querytype..";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

%mend lsaf_getquerycolumns;