%* lsaf_getproperties                                                                              *;
%*                                                                                                 *;
%* Gets the properties of an item in the repository. The properties are stored in a SAS data set.  *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 1.2                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2013-08-30  updates for standardization and coding conventions                                  *;
%* 2014-04-18  updates to surface extended attribute metadata.                                     *;
%* 2016-08-26  rebrand as LSAF                                                                     *;
%* 2017-01-10  updates to retrieve a file by version number                                        *;
%* 2018-12-10  documentation update for LSAF 5.x behavior                                          *;
%* 2019-11-01  update to handle a blank SAS output file.                                           *;
%* 2021-08-13  Remove header documentation                                                         *;
%* 2021-10-12  US11739                                                                             *;
%*                                                                                                 *;

%macro lsaf_getproperties(
  lsaf_path=,
  lsaf_version=,
  sas_dsname=work.lsafGetProperties
  ) /des='Get a list of properties for a specified item in the SAS Life Science Analytics Framework Repository.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetProperties;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local notesOption
         _lsafRowCount_
  ;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=path name displayName value type isReadonly isExtended);
    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300
           rowCount        length=8

           path            length=$2048    label="Item Path"
           name            length=$256     label="Property Name"
           displayName     length=$256     label="Property Display Name"
           value           length=$2048    label="Property Value"
           type            length=$32      label="Property Type"
           isReadonly      length=8        label="Property value is not modifiable."
           isExtended      length=8        label="Property is an Extended Attribute"
   ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/content/repository/RepositoryService") ;
    declare javaobj results("com/sas/lsaf/macro/content/repository/SasRepositoryItemPropertiesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getProperties", "%bquote(&lsaf_path)", "%bquote(&lsaf_version)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of properties for &lsaf_path" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of properties were returned for &lsaf_path" ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        path="&lsaf_path";
        results.callStringMethod("getPropertyName", row, name);
        results.callStringMethod("getPropertyValue", row, value);
        results.callStringMethod("getPropertyDisplayName", row, displayName);
        results.callStringMethod("getPropertyAttributeType", row, type);
        results.callBooleanMethod("getPropertyIsReadOnly", row, isReadonly);
        results.callBooleanMethod("getPropertyIsExtended", row, isExtended);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No properties were retrieved.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by name;
    run;
    options &notesOption;
  %end;

%mend lsaf_getproperties ;
